<?php

/*

  LUNAR HEMISPHERIC PHASE EXPLORERS - HD EDITION FOR LARGE MONITORS

  Original images rendered at 2560x1920

  These simple programs display the current lunar ephemeris statistics
  and simultaneous phase images as viewed from four lunar hemispheres
  at the current date and time (UT) with the option of viewing the
  hemispheres at any relative phase angle from 0 to 360 degrees.

  NOTE:
  The images have been converted to WEBP format to conserve bandwidth.


   Author   : Jay Tanner - Revised: 2017-2025
   Language : PHP v5.6.11 - Revised with PHP 8.2.12
   License  : Source code and images released to the public domain.
*/

// Get current year.
   $cYear = GMDate("Y");

   $_AUTHOR_          = 'Jay Tanner';
   $_PROGRAM_VERSION_ = ''; $at = "&#97;&#116;"; $UTC = "&#85;&#84;&#67";
   $_SCRIPT_PATH_     = Filter_Input(INPUT_SERVER, 'SCRIPT_FILENAME');
   $_REVISION_DATE_   = $_PROGRAM_VERSION_ .'Revised: '. gmdate("l, F d, Y $at h:i:s A   $UTC", FileMTime($_SCRIPT_PATH_));

// Construct download link so that it only shows up
// when the program is posted on the public WWW. On
// the local desktop, the download link is supressed.
   $DLLink = '';
   if(substr($_SCRIPT_PATH_, 0,3) <> 'D:/')
  {
   $DLLink = "<tr><td colspan='2'><a href='Lunar-Hemispheric-Phase-Explorers-HD.7z' style='color:DodgerBlue; text-decoration:none; font-family:Verdana; font-size:100%; font-weight:normal;'>Download PHP Source Code and Graphics For This Program</a><br><span style='color:orange; font-family:Verdana; font-size:90%;'>Approx: 1.14 GB - Public Domain</span><br></td></tr>";
  }

// exit("$_REVISION_DATE_");

   $ColSpan = 6; // Same as number of input variables.

// --------------------------------------------
// Get current date/time (UTC) for computations
// and construct Date/Time string argument.

   $DateString = GMDate("Ymd");   // UD
   $TimeString = GMDate("H:i:s"); // UTC
   $DateTimeString = "$DateString $TimeString";

// --------------------------------------------------------
// Get current date (UT), spelled out in full, for display.
   $FullDateString = GMDate("Y M d - D");





// --------------------------------------------------------------
// Compute JDTT for current UT by applying NASA Delta T estimate.
// Then, round and format the result to 8 decimals.
// FALSE = Time is TT = Default
// TRUE  = Time is UT

   $JDTT = SPrintF("%7.8f", Date_Time_Zone_to_JD_Zone($DateTimeString, TRUE));

// ---------------------------------------
// Compute applied Delta T resolved to the
// nearest second in -/+ HH:MM:SS format.
// From NASA polynomial expressions.

   $DeltaTHMS = Delta_T_HMS($DateTimeString);

// --------------------------------------------------
// Compute current ephemeris statistics for the moon.

   $BodyStatsString = Geocentric_Ephem_Stats("Moon", $JDTT);

// -------------------------------------------------
// Extract the geocentric lunar ephemeris statistics
// from the string into their respective variables.

   list ($RAHrs, $DeclDeg, $DistKm, $PhaseAngDeg)
   = preg_split("[ ]", $BodyStatsString);

// ------------------------------------------------------------------
// Compute current relative phase angles for each side to the nearest
// 1/2 degree. The normal near side geocentric phase is computed first
// and then the relative phase angles from the other hemispheres are
// then derived from this value.

// Near side.
   $PhaseAngNear = round($PhaseAngDeg, 0);
   if ($PhaseAngNear == 360) {$PhaseAngNear = 0;}
   $PhaseIMGNear = SPrintF("%03d", $PhaseAngNear);


// West hemisphere.
   $PhaseAngEast = $PhaseAngNear + 270;
   $PhaseAngEast -= ($PhaseAngEast > 360)? 360:0;
   if ($PhaseAngEast == 360) {$PhaseAngEast = 0;}
   $PhaseIMGEast = SPrintF("%03d", $PhaseAngEast);

// East hemisphere.
   $PhaseAngWest = $PhaseAngNear + 90;
   $PhaseAngWest -= ($PhaseAngWest > 360)? 360:0;
   if ($PhaseAngWest == 360) {$PhaseAngWest = 0;}
   $PhaseIMGWest = SPrintF("%03d", $PhaseAngWest);



// Far side.
   $PhaseAngFar = $PhaseAngNear + 180;
   $PhaseAngFar -= ($PhaseAngFar > 360)? 360:0;
   if ($PhaseAngFar == 360) {$PhaseAngFar = 0;}
   $PhaseIMGFar = SPrintF("%03d", $PhaseAngFar);

// ----------------------------------
// Format computed values for output.

   $JDTT    = SPrintF("%7.8f", $JDTT);
   $RAHMS   = Hours_to_HMS($RAHrs, 3);
   $DeclDMS = Deg_to_DMS($DeclDeg, 2, TRUE);
   $DistMi  = SPrintF("%6.3f", $DistKm/1.609344);
   $DistKm  = SPrintF("%6.3f", $DistKm);


   $PhaseAngDeg = SPrintF("%1.3f", $PhaseAngDeg);

// ----------------------------------
// Print out ephemeris statistics and
// display current lunar phase images
// as viewed from 4 hemispheres.



   print
"
<!DOCTYPE HTML>
<html lang='en'>
<!-- Google tag (gtag.js) -->
<script async src='https://www.googletagmanager.com/gtag/js?id=G-WJ57EQVV4C'></script>
<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());

  gtag('config', 'G-WJ57EQVV4C');
</script>

<head>
<title>Current Lunar Phases as Viewed From Four Sides</title>

<meta http-equiv='content-type' content='text/html; charset=UTF-8'>
<meta name='description' content='View the entire lunar surface at any relative phase.'>
<meta name='keywords' content='lunar phase,lunar phases,lunar phase explorers,moon phase'>
<meta name='author' content='Jay Tanner - PHPScienceLabs.com'>
<meta http-equiv='pragma'  content='no-cache'>
<meta http-equiv='expires' content='-1'>
<meta name='robots'    content='index,follow'>
<meta name='googlebot' content='index,follow'>

<style type='text/css'>
TABLE {background-color:black; font-size:100%; border:1px solid #444444;}
TD {color:#EFEFEF; font-family:monospace; padding:4px; text-align:center; border:1px solid #222222;}
PRE {font-family:monospace; font-size:110%; text-align:left;}
</style>

</head>

<body style='background-color:black;'>

<div>
<table align='left'>
<tr>
<td style='text-align:center; color:white; background-color:#000066; border:2px solid white; font-family:Verdana; font-weight:normal; font-size:120%; border-radius:8px 8px 0px 0px; line-height:125%;' colspan='3'>
Lunar Hemispheric Phase Explorers - HD Edition
<br><br>
Current Lunar Phases as Viewed From Above the Four Hemispheres<br>
<span style='font-size:90%;'>Near Side, Far Side, Western and Eastern Hemispheres
<br><br>
<b style='font-weight:normal; font-size:80%;'>Program and Graphics by Jay Tanner - $cYear</b></span><br>
</td></tr>


<td colspan='2'>
<pre style='color:#E8E8E8; background:#002200; font-size:150%; font-weight:normal; padding:4px; text-align:left; border:2px solid lime; border-radius:0px 0px 8px 8px;'>
              Current Geocentric Lunar Coordinates, Distance and Phase &nbsp;
                      Based on the NASA/JPL DE405 Ephemeris Model

                        Working Date UD    =  $FullDateString
                        Working Time UT    =  $TimeString
                        Astronomical JD TT =  $JDTT

                  Right Ascension = $RAHrs hr = $RAHMS
                  Declination     = $DeclDeg deg = $DeclDMS
                  Distance        = $DistKm km = $DistMi mi
                  Phase Angle     = $PhaseAngDeg deg

</pre>
<span style='font-family:Verdana; font-size:85%; font-weight:normal; color:white;'>REFRESH Page to Update Statistics and Images to Current Date and Time<br>&nbsp;</span><br><span style='font-family:Verdana; font-size:85%; font-weight:normal; color:white;'>CLICK On Any Hemispheric Image For a Full HD View In a New Tab</span>
</td>



<tr>
<td colspan='1' style='color:silver; background:#000012; padding:4px; text-align:center; border:2px solid #444444; border-radius:8px;'>
<b><span style='font-size:100%; font-family:Verdana; font-weight:normal; font-size:100%; color:silver;'>N</span><br><a href='lunar-near-side-phase-explorer/' target='_blank' ><img src=\"near/$PhaseIMGNear.webp\" title=' Click Image For Full HD View In a New Tab ' alt=''></a><br><br>
<span style='font-size:9pt; font-weight:normal; color:white; font-family:Verdana;'>Current&nbsp;Near&nbsp;Side&nbsp;View<br>Standard&nbsp;Geocentric&nbsp;Earth&nbsp;View<br>Sans&nbsp;Librations<br>Geocentric&nbsp;Phase:&nbsp;$PhaseAngNear&deg;</span></b><br><br>
</td>


<td colspan='1' style='color:silver; background:black; padding:4px; text-align:center; border:2px solid #444444; border-radius:8px;'>
<b><span style='font-size:9pt; font-family:Verdana; font-weight:normal; color:gray;'>N</span><br><a href='lunar-far-side-phase-explorer/' target='_blank' ><img src=\"far/$PhaseIMGFar.webp\" title=' Click Image For Full HD View In a New Tab ' alt=''></a><br><br>
<span style='font-size:9pt; font-weight:normal; color:silver; font-family:Verdana;'>Current&nbsp;Far&nbsp;Side&nbsp;View<br>Behind&nbsp;the&nbsp;Moon&nbsp;Looking&nbsp;Back<br>Earth&nbsp;is&nbsp;Directly&nbsp;Behind&nbsp;Moon<br>Relative&nbsp;Phase:&nbsp;$PhaseAngFar&deg;</span></b><br><br>
</td>
</tr>


<tr>
<td style='color:silver; background-color:black; padding:4px; text-align:center; border:2px solid #444444; border-radius:8px;'>
<b><span style='font-size:9pt; font-family:Verdana; font-weight:normal; color:gray;'>N</span><br><a href='lunar-west-side-phase-explorer/' target='_blank' ><img src=\"east/$PhaseIMGEast.webp\" title=' Click Image For Full HD View In a New Tab ' alt=''></a><br><br>
<span style='font-size:9pt; font-weight:normal; color:silver; font-family:Verdana; '>Current&nbsp;West&nbsp;(Left)&nbsp;Side&nbsp;View<br>Centered&nbsp;Over&nbsp;Western&nbsp;Hemisphere<br>Earth&nbsp;is&nbsp;Directly&nbsp;to&nbsp;the&nbsp;Right<br>Relative&nbsp;Phase:&nbsp;$PhaseAngEast&deg;</span></b><br><br>
</td>


<td style='color:silver; background-color:black; padding:4px; text-align:center; border:2px solid #444444; border-radius:8px;'>
<b><span style='font-size:9pt; font-family:Verdana; color:gray; font-weight:normal;'>N</span><br><a href='lunar-east-side-phase-explorer/' target='_blank' ><img src=\"west/$PhaseIMGWest.webp\" title=' Click Image For Full HD View In a New Tab ' alt=''></a><br><br>
<span style='font-size:9pt; font-weight:normal; color:silver; font-family:Verdana;'>Current&nbsp;East&nbsp;(Right)&nbsp;Side&nbsp;View<br>Centered&nbsp;Over&nbsp;Eastern&nbsp;Hemisphere<br>Earth&nbsp;is&nbsp;Directly&nbsp;to&nbsp;the&nbsp;Left<br>Relative&nbsp;Phase:&nbsp;$PhaseAngWest&deg;</span></b>
<br><br>
</td>
</tr>


<tr><td colspan='2'>&nbsp;</td></tr>
$DLLink
<tr><td colspan='2'>
<textarea rows='86' cols='81' ReadOnly style='font-size:13.5pt; background:#002200; color:white; font-weight:normal; border:2px solid lime; border-radius:8px; padding:8px;'>
         LUNAR HEMISPHERIC PHASE EXPLORERS - HD EDITION FOR LARGE MONITORS
                DEVELOPED USING AN ULTRA-WIDE (2560x1080) HD MONITOR

Author   : Jay Tanner
Revised  : 2017-2025
Language : PHP v5.6.11 - v8.2.12

All lunar images in this program were rendered by LUNEX For POV-Ray v3.7.

POV-Ray is the free, open-source 3D ray-tracing software used to render the
lunar phase images.         https://www.povray.org

##############################################################################

This program displays shaded lunar phase simulations of the four lunar
hemispheres based on the NASA/JPL DE405 ephemeris model and lunar surface
map composed from Clementine imaging and a digital elevation model (DEM)
derived from the Lunar Orbiter Laser Altimeter (LOLA) altimetry data.

On initial startup, the current phases are computed using the NASA/JPL DE405
ephemeris model spanning the 600-year period from 1600 to 2200, referred to
the true equinox and ecliptic of the date. Refreshing the page updates the
geocentric phases and ephemeris statistics for the current date and UTC.

Although the images appear like photographs, they are computer-generated
simulations based on NASA data and the phase shading computed from actual
solar and lunar positions at the moment of computation, simulating reality
very closely.

There are 4 sets of lunar surface maps centered on the front side, far side,
eastern and western hemispheres which can be viewed at any relative phase
from 0 to 360 degrees.  The images are 1500x1500 PNG images on transparent
backgrounds covering the entire lunar surface.

In terms of 3D perspective, all the hemispheric phase images are as viewed
from the same mean geocentric distance of 238857 mi (384403 km) through a
virtual telescope.

Imagine the moon at the center of a circle with the sun moving around it
from 0 to 360 degrees in a clockwise direction.  The position of the sun
on this phase circle is the simple lunar phase angle. The new moon is at
zero degrees with the sun behind the moon, progressing clockwise from
there and cycling through all the lunar phases from 0 to 360 degrees.

Given the geocentric ecliptical longitudes of the sun and moon at the same
moment, the simple lunar phase angle at the same moment can be computed to
sufficient visual precision from the simple algorithm given below.


                      A SIMPLE LUNAR PHASE ANGLE ALGORITHM
            --------------------------------------------------------
            Let:

            Ls = Geocentric Ecliptical Longitude Of Sun  (0 to 360&deg;)
            Lm = Geocentric Ecliptical Longitude Of Moon (0 to 360&deg;)
            PA = Geocentric Lunar Phase Angle (0 to 360&deg;)

            Then:

            a = 360 − Lm + Ls
            if (a > 360) {a = a − 360}
            PA = 360 − a

            or:

            a = 360 − Lm + Ls
            PA = 360 − a −= (a > 360)? 360:0
            --------------------------------------------------------


                     RELATIVE PHASE ANGLES (NORTH IS UPWARD)
            --------------------------------------------------------
              Relative Phase      PA       Direction of Illumination
            ------------------  -------    -------------------------
               New Phase        0&#176; / 360&#176;  From behind the moon
               First Quarter      90&#176;      Directly from the right
               Full Phase         180&#176;     From behind the eye
               Last Quarter       270&#176;     Directly from the left
            ---------------------------------------------------------
</textarea>

</td></tr>

<tr><td colspan='$ColSpan' style='font-family:Verdana; color:gray; background-color:black;'>Program and Graphics by $_AUTHOR_
</td></tr>

</table>
<br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br>
</div>
</body>
</html>
";




// *****************
// *****************

/*
   A101.php

   THIS ASTRONOMY MODULE CONTAINS THE COMMON CORE
   LOGIC FOR THE LUNAR HEMISPHERIC PHASE EXPLORERS.
*/



// -----------------------------------------------------------------------
// This function strips all leading, trailing and internal whitespace from
// within a text string.  All occurrences of multiple spaces within the
// text string will be replaced by single spaces and all leading and
// trailing whitespace will also be removed.

   function Strip_WSpace ($TextString)
{
   return preg_replace("/\s+/", " ", trim($TextString));
}


/*
   ---------------------------------------
   This function simply swaps the contents
   of two simple variable arguments.
*/

   function Swap (&$var1, &$var2)
 {
   $w=$var1;  $var1=$var2;  $var2=$w;
 }



/*
   =========================================================================
   Level(1)

   This function serves as an inverse JDTT function.  Given a JDTT argument
   and local time zone offset, this function returns the local date and time
   resolved to the nearest second.

   NOTE:
   The optional Delta T value is resolved to the nearest second.
   =========================================================================
*/

   function JD_Zone_to_Date_Time_Zone ($JDTT_TZ, $dTFlag=FALSE)
{
// Strip all redundant white space from JDTT/zone string
// and copy into temporary working variable (w).
   $w = strip_wspace($JDTT_TZ);

// If no time zone given, then set default = UT.
   if (strpos($w, " ") === FALSE) {$w .= " +00:00";}

   list($JDTT, $TZhhmm) = preg_split("[ ]", $w);

// Tidy up Time Zone string.
   $TZhhmm = Hours_to_HMS(HMS_to_Hours($TZhhmm), 0, TRUE, ':');
// ($hours, $ssDecimals=0, $PosSignFlag=FALSE, $SymbolsMode=2)

// Compute dT as fraction of a day.
   $TZFrac = HMS_to_Hours($TZhhmm)/24;

// Compute JD12TT value for date corresponding to JDTT argument.
   $JD12TT = floor($JDTT + 0.5);

// Compute Gregorian date corresponding to the JD argument.
   list($m, $d, $Y) = preg_split("[\/]", JDtoGregorian($JD12TT));

// Optionally apply Delta T (Default dTFlag === TRUE).
    $dTsec = $dTFrac = 0;

   if ($dTFlag === TRUE)
{
   $dTseconds = Delta_T ($Y*100 + $m);         // Raw value in seconds.
   $dTsec  = SPrintF("%+1.0f", $dTseconds);    // To nearest second.
   $dTFrac = SPrintF("%+1.16f", $dTsec/86400); // As fraction of a day.
}

// Compute JDUT by subtracting dTFrac from JDTT.
   $JDUT = SPrintF("%1.8f", $JDTT - $dTFrac);

// Compute JDLT for local time zone from JDUT.
   $JDLT = SPrintF("%1.8f", $JDUT + $TZFrac);

// Compute JD12LT for local standard date and time.
   $JD12LT = floor($JDLT + 0.5);

// Compute Gregorian date corresponding to the JDLT.
   list($m, $d, $Y) = preg_split("[\/]", JDtoGregorian($JD12LT));

// Force month and day values into 2-digit format.
   $m = SPrintF("%02d", $m);
   $d = SPrintF("%02d", $d);

// Construct integer encoded date from date elements.
   $Ymd = "$Y$m$d";

// Convert fractional part of JDLT into hours local time.
   $j = bcsub($JDLT, "0.5", 16);
   $hours = bcmul(24, "0" . substr($j, strpos($j, "."), strlen($j)), 16);

// Compute local time elements (hh,mm,ss).
   $hh      = floor($hours);
   $minutes = 60*($hours - $hh);
   $mm      = floor($minutes);
   $seconds = 60*($minutes - $mm);

// Format (hh, mm, ss) values for output.
// ss is rounded to nearest second.
   $hh = SPrintF("%02d",  $hh);
   $mm = SPrintF("%02d",  $mm);
   $ss = SPrintF("%02d",  $seconds + 0.5);

// Patch for that blasted 60 glitch.
   if ($ss == 60){$ss = "00";  $mm = SPrintF("%02d",  $mm+1);}
   if ($mm == 60){$mm = "00";  $hh = SPrintF("%02d",  $hh+1);}

// Remove seconds part of TZ because
// it always equates to zero anyway.
   $w = "$Ymd $hh:$mm:$ss $TZhhmm";
   $w = substr($w, 0, strlen($w)-3);

// Done.
   return $w;

} // End of  JD_Zone_to_Date_Time_Zone(...)





/*
   ===================================================================
   Level(0)

   This function computes the local or Greenwich mean sidereal time in
   degrees for any given JD and geographic longitude arguments.

   ARGUMENTS:
   JDTT    = JD argument for any given date and time (TT)

   LngDeg  = Geographic longitude. (Negative = West)
             Default longitude = 0 degrees (= Greenwich)

   ERRORS:
   FALSE is returned if either argument is invalid.
   ===================================================================
*/

   function Mean_Sid_Time ($JDTT, $LngDeg=0)
{
// ------------------------------
// Check for invalid argument(s).
   if ($JDTT < 2305447 or $JDTT > 2524959 or !is_numeric($LngDeg)) {return FALSE;}

// ------------------------------------------------
// Compute time (t) in Julian millennia as reckoned
// from J2000.0 and corresponding to the JD number.
   $t = ($JDTT - 2451545) / 36525;

// ------------------------------------------------
// Compute Greenwich mean sidereal time in degrees.
   $GSTMeanDeg = 280.46061837 + 360.98564736629*($JDTT - 2451545)
               + ($t*$t * 0.000387993) - ($t*$t*$t / 38710000);

// -----------------------------------
// Compute local mean sidereal time at
// the given geographic longitude.
   $LSTMeanDeg = $GSTMeanDeg + $LngDeg;

// ----------------------------------------------------------------
// Modulate sidereal time angle to fall between 0 and +360 degrees.
   $w = abs($LSTMeanDeg);
   $w = (($LSTMeanDeg < 0)? -1:1)*($w - 360*floor($w/360));
   $w += ($w < 0)? 360:0;
   $LSTMeanDeg = $w;
   $LSTMeanHrs = $LSTMeanDeg / 15;

// -------------------------------------------
// Return mean sidereal time angle in degrees.
   return rtrim(rtrim($LSTMeanDeg,'0'), '.');

}  // End of  Mean_Sid_Time (...)








/*
  ========================================================================
  This function computes the mean obliquity of the ecliptic in degrees
  for the current JDTT argument corresponding to any given date/time.

  This ecliptic obliquity function is based on a formula derived by
  J. Laskar, Astronomy and Astrophysics, 1968, Vol. 157, p68.

  The estimated accuracy of this formula is ±0.01 arc second between
  the years 1000 AD and 3000 AD and a few arc seconds after 10,000
  years.  It is only valid in the range ±10,000 years either way
  of J2000.0

  Over the long term, it is more accurate than the formula adopted by
  the International Astronomical Union (IAU).


  ---------
  ARGUMENT:
  $JDTT = JD number at the moment (TDB) for which the mean obliquity
          of the ecliptic is required.
  ========================================================================
*/

   function Mean_Eps ($JDTT)
{

// ----------------------------------------------
// Compute the time in Julian centuries reckoned
// from J2000.0 corresponding to the JD argument
// and then divide by 100 for decamillennia.

   $t = ($JDTT - 2451545) / 36525 / 100;

// ------------------------------------------
// Compute mean obliquity (e) of the ecliptic
// expressed in arc seconds.

   $p = $t*$t;

   $e = 84381.448 - 4680.93*$t;

   $e -=    1.55*$p;  $p *= $t;
   $e += 1999.25*$p;  $p *= $t;
   $e -=   51.38*$p;  $p *= $t;
   $e -=  249.67*$p;  $p *= $t;
   $e -=   39.05*$p;  $p *= $t;
   $e +=    7.12*$p;  $p *= $t;
   $e +=   27.87*$p;  $p *= $t;
   $e +=    5.79*$p;  $p *= $t;
   $e +=    2.45*$p;

// -----------------------------------
// Done.  Return obliquity in degrees.
   return $e / 3600;

} // End of  Mean_Eps (...)






/*
  ========================================================================
  This function computes the nutation in obliquity (delta epsilon)
  in degrees according to the IAU 2000B nutation series.

  This value is applied to the mean obliquity of the ecliptic to
  obtain the true, apparent obliquity.

  The series consists of 77 terms.

  ARGUMENT:
  $JD = JD number at the moment (TDB) for which the
        nutation in obliquity is required.

  ERRORS:
  No error checking.

  ========================================================================
*/


   function dEps_2000B($JDTT)
{

   $T = ($JDTT - 2451545) / 36525;

   $T2 = $T * $T;
   $T3 = $T * $T2;
   $T4 = $T * $T3;

// -------------------------------------------
// Compute mean anomaly of the Moon in radians

   $L  = deg2rad((485868.249036 + 1717915923.2178*$T + 31.8792*$T2
       + 0.051635*$T3 - 0.0002447*$T4) / 3600);


// ------------------------------------------
// Compute mean anomaly of the Sun in radians

   $Lp = deg2rad((1287104.79305 + 129596581.0481*$T
       - 0.5532*$T2  + 0.000136*$T3 - 0.00001149*$T4) / 3600);


// ------------------------------------------------------------
// Compute mean argument of the latitude of the Moon in radians

   $F  = deg2rad((335779.526232 + 1739527262.8478*$T
       - 12.7512*$T2 - 0.001037*$T3 + 0.00000417*$T4) / 3600);


// -----------------------------------------------------------
// Compute mean elongation of the Moon from the Sun in radians

   $D  = deg2rad((1072260.70369 + 1602961601.2090*$T
       - 6.3706*$T2  + 0.006593*$T3 - 0.00003169*$T4) / 3600);


// ------------------------------------------------------------------------
// Compute mean longitude of the mean ascending node of the Moon in radians

   $Om = deg2rad((450160.398036 - 6962890.5431*$T
       + 7.4722*$T2  + 0.007702*$T3 - 0.00005939*$T4) / 3600);

// --------------------------------------------------------------
// Sum series for nutation in obliquity (dEps) in arc sec * 10E+7

$s = 0;
$s += (92052331 + 9086*$T)*cos($Om) + 15377*sin($Om);
$s += (5730336 - 3015*$T)*cos(2*($F - $D + $Om)) - 4587*sin(2*($F - $D + $Om));
$s += (978459 - 485*$T)*cos(2*($F + $Om)) + 1374*sin(2*($F + $Om));
$s += (-897492 + 470*$T)*cos(2*$Om) - 291*sin(2*$Om);
$s += (73871 - 184*$T)*cos($Lp) - 1924*sin($Lp);
$s += (224386 - 677*$T)*cos($Lp + 2*($F - $D + $Om)) - 174*sin($Lp + 2*($F - $D + $Om));
$s -= 6750*cos($L) - 358*sin($L);
$s += (200728 + 18*$T)*cos(2*$F + $Om) + 318*sin(2*$F + $Om);
$s += (129025 - 63*$T)*cos($L + 2*($F + $Om)) + 367*sin($L + 2*($F + $Om));
$s += (-95929 + 299*$T)*cos(2*($F - $D + $Om) - $Lp) + 132*sin(2*($F - $D + $Om) - $Lp);
$s += (-68982 - 9*$T)*cos(2*($F - $D) + $Om) + 39*sin(2*($F - $D) + $Om);
$s += (-53311 + 32*$T)*cos(2*($F + $Om) - $L) - 4*sin(2*($F + $Om) - $L);
$s -= 1235*cos(2*$D - $L) - 82*sin(2*$D - $L);
$s -= 33228*cos($L + $Om) + 9*sin($L + $Om);
$s += 31429*cos($Om - $L) - 75*sin($Om - $L);
$s += (25543 - 11*$T)*cos(2*($F + $D + $Om) - $L) + 66*sin(2*($F + $D + $Om) - $L);
$s += 26366*cos($L + 2*$F + $Om) + 78*sin($L + 2*$F + $Om);
$s += (-24236 - 10*$T)*cos(2*($F - $L) + $Om) + 20*sin(2*($F - $L) + $Om);
$s -= 1220*cos(2*$D) - 29*sin(2*$D);
$s += (16452 - 11*$T)*cos(2*($F + $D + $Om)) + 68*sin(2*($F + $D + $Om));
$s -= 13870*cos(-2*($Lp - $F + $D - $Om));
$s += 477*cos(2*($D - $L)) - 25*sin(2*($D - $L));
$s += (13238 - 11*$T)*cos(2*($L + $F + $Om)) + 59*sin(2*($L + $F + $Om));
$s += (-12338 + 10*$T)*cos($L + 2*($F - $D + $Om)) - 3*sin($L + 2*($F - $D + $Om));
$s -= 10758*cos(2*$F + $Om - $L) + 3*sin(2*$F + $Om - $L);
$s -= 609*cos(2*$L) - 13*sin(2*$L);
$s -= 550*cos(2*$F) - 11*sin(2*$F);
$s += (8551 - 2*$T)*cos($Lp + $Om) - 45*sin($Lp + $Om);
$s -= 8001*cos(-$L + 2*$D + $Om) + sin(-$L + 2*$D + $Om);
$s += (6850 - 42*$T)*cos(2*($Lp + $F - $D + $Om)) - 5*sin(2*($Lp + $F - $D + $Om));
$s -= 167*cos(2*($D - $F)) - 13*sin(2*($D - $F));
$s += 6953*cos($L - 2*$D + $Om) - 14*sin($L - 2*$D + $Om);
$s += 6415*cos($Om - $Lp) + 26*sin($Om - $Lp);
$s += 5222*cos(2*($F + $D) + $Om - $L) + 15*sin(2*($F + $D) + $Om - $L);
$s += (168 - $T)*cos(2*$Lp) + 10*sin(2*$Lp);
$s += 3268*cos($L + 2*($F + $D + $Om)) + 19*sin($L + 2*($F + $D + $Om));
$s += 104*cos(2*($F - $L)) + 2*sin(2*($F - $L));
$s -= 3250*cos($Lp + 2*($F + $Om)) + 5*sin($Lp + 2*($F + $Om));
$s += 3353*cos(2*($F + $D) + $Om) + 14*sin(2*($F + $D) + $Om);
$s += 3070*cos(2*($F + $Om) - $Lp) + 4*sin(2*($F + $Om) - $Lp);
$s += 3272*cos(2*$D + $Om) + 4*sin(2*$D + $Om);
$s -= 3045*cos($L + 2*($F - $D) + $Om) + sin($L + 2*($F - $D) + $Om);
$s -= 2768*cos(2*($L + $F - $D + $Om)) + 4*sin(2*($L + $F - $D + $Om));
$s += 3041*cos(2*($D - $L) + $Om) - 5*sin(2*($D - $L) + $Om);
$s += 2695*cos(2*($L + $F) + $Om) + 12*sin(2*($L + $F) + $Om);
$s += 2719*cos(2*($F - $D) + $Om - $Lp) - 3*sin(2*($F - $D) + $Om - $Lp);
$s += 2720*cos($Om - 2*$D) - 9*sin($Om - 2*$D);
$s -= 51*cos(-$L - $Lp + 2*$D) - 4*sin(-$L - $Lp + 2*$D);
$s -= 2206*cos(2*($L - $D) + $Om) - sin(2*($L - $D) + $Om);
$s -= 199*cos($L + 2*$D) - 2*sin($L + 2*$D);
$s -= 1900*cos($Lp + 2*($F - $D) + $Om) - sin($Lp + 2*($F - $D) + $Om);
$s -= 41*cos($L - $Lp) - 3*sin($L - $Lp);
$s += 1313*cos(-2*($L - $F - $Om)) - sin(-2*($L - $F - $Om));
$s += 1233*cos(3*$L + 2*($F + $Om)) + 7*sin(3*$L + 2*($F + $Om));
$s -= 81*cos(-$Lp + 2*$D) - 2*sin(-$Lp + 2*$D);
$s += 1232*cos($L - $Lp + 2*($F + $Om)) + 4*sin($L - $Lp + 2*($F + $Om));
$s -= 20*cos($D) + 2*sin($D);
$s += 1207*cos(2*($F + $D + $Om) - $L - $Lp) + 3*sin(2*($F + $D + $Om) - $L - $Lp);
$s += 40*cos(2*$F - $L) - 2*sin(2*$F - $L);
$s += 1129*cos(-$Lp + 2*($F + $D + $Om)) + 5*sin(-$Lp + 2*($F + $D + $Om));
$s += 1266*cos($Om - 2*$L) - 4*sin($Om - 2*$L);
$s -= 1062*cos($L + $Lp + 2*($F + $Om)) + 3*sin($L + $Lp + 2*($F + $Om));
$s -= 1129*cos(2*$L + $Om) + 2*sin(2*$L + $Om);
$s -= 9*cos($Lp + $D - $L);
$s += 35*cos($L + $Lp) - 2*sin($L + $Lp);
$s -= 107*cos($L + 2*$F) - sin($L + 2*$F);
$s += 1073*cos(2*($F - $D) + $Om - $L) - 2*sin(2*($F - $D) + $Om - $L);
$s += 854*cos($L + 2*$Om);
$s -= 553*cos($D - $L) + 139*sin($D - $L);
$s -= 710*cos(2*($F + $Om) + $D) + 2*sin(2*($F + $Om) + $D);
$s += 647*cos(2*($F + 2*$D + $Om) - $L) + 4*sin(2*($F + 2*$D + $Om) - $L);
$s -= 700*cos($Lp + $D + $Om - $L);
$s += 672*cos(-2*($Lp - $F + $D) + $Om);
$s += 663*cos($L + 2*($F + $D) + $Om) + 4*sin($L + 2*($F + $D) + $Om);
$s -= 594*cos(-2*($L - $F - $D - $Om)) + 2*sin(-2*($L - $F - $D - $Om));
$s -= 610*cos(2*$Om - $L) - 2*sin(2*$Om - $L);
$s -= 556*cos($L + $Lp + 2*($F - $D + $Om));

// ----------------------------
// Return nutation in obliquity
// expressed in degrees.

   return $s / 36000000000;


} // End of  dEps_2000B(...)




/*

   ========================================================================
   This function computes the nutation in ecliptical longitude (Delta psi)
   in degrees according to the IAU 2000B nutation series.

   This value is applied to the mean ecliptical longitude to obtain the
   true, apparent ecliptical longitude.

   The series consists of 77 terms.

   ARGUMENT:
   $JDTT = Astronomical JD number corresponding to date and time (TT).

   ERRORS:
   No error checking.
   ========================================================================
*/

   function dPsi_2000B ($JDTT)
{

// ---------------------------------------------
// Compute time (T) in Julian centuries reckoned
// from J2000.0 and 4 powers of T.

   $T  = ($JDTT - 2451545) / 36525;

   $T2 = $T * $T;
   $T3 = $T * $T2;
   $T4 = $T * $T3;

// -------------------------------------------
// Compute mean anomaly of the Moon in radians

   $L  = deg2rad((485868.249036 + 1717915923.2178*$T + 31.8792*$T2
       + 0.051635*$T3 - 0.0002447*$T4) / 3600);


// ------------------------------------------
// Compute mean anomaly of the Sun in radians

   $Lp = deg2rad((1287104.79305 + 129596581.0481*$T
       - 0.5532*$T2  + 0.000136*$T3 - 0.00001149*$T4) / 3600);


// ------------------------------------------------------------
// Compute mean argument of the latitude of the Moon in radians

   $F  = deg2rad((335779.526232 + 1739527262.8478*$T
       - 12.7512*$T2 - 0.001037*$T3 + 0.00000417*$T4) / 3600);


// -----------------------------------------------------------
// Compute mean elongation of the Moon from the Sun in radians

   $D  = deg2rad((1072260.70369 + 1602961601.2090*$T
       - 6.3706*$T2  + 0.006593*$T3 - 0.00003169*$T4) / 3600);


// ------------------------------------------------------------------------
// Compute mean longitude of the mean ascending node of the Moon in radians

   $Om = deg2rad((450160.398036 - 6962890.5431*$T
       + 7.4722*$T2  + 0.007702*$T3 - 0.00005939*$T4) / 3600);

// -----------------------------------------------------------------------
// Sum 2000B series for nutation in longitude (dPsi) in arc sec * 10000000

$s = 0;
$s += (-172064161 - 174666*$T)*sin($Om) + 33386*cos($Om);
$s += (-13170906 - 1675*$T)*sin(2*($F - $D + $Om)) - 13696*cos(2*($F - $D + $Om));
$s += (-2276413 - 234*$T)*sin(2*($F + $Om)) + 2796*cos(2*($F + $Om));
$s += (2074554 + 207*$T)*sin(2*$Om) - 698*cos(2*$Om);
$s += (1475877 - 3633*$T)*sin($Lp) + 11817*cos($Lp);
$s += (-516821 + 1226*$T)*sin($Lp + 2*($F - $D + $Om)) - 524*cos($Lp + 2*($F - $D + $Om));
$s += (711159 + 73*$T)*sin($L) - 872*cos($L);
$s += (-387298 - 367*$T)*sin(2*$F + $Om) + 380*cos(2*$F + $Om);
$s += (-301461 - 36*$T)*sin($L + 2*($F + $Om)) + 816*cos($L + 2*($F + $Om));
$s += (215829 - 494*$T)*sin(2*($F - $D + $Om) - $Lp) + 111*cos(2*($F - $D + $Om) - $Lp);
$s += (128227 + 137*$T)*sin(2*($F - $D) + $Om) + 181*cos(2*($F - $D) + $Om);
$s += (123457 + 11*$T)*sin(2*($F + $Om) - $L) + 19*cos(2*($F + $Om) - $L);
$s += (156994 + 10*$T)*sin(2*$D - $L) - 168*cos(2*$D - $L);
$s += (63110 + 63*$T)*sin($L + $Om) + 27*cos($L + $Om);
$s += (-57976 - 63*$T)*sin($Om - $L) - 189*cos($Om - $L);
$s += (-59641 - 11*$T)*sin(2*($F + $D + $Om) - $L) + 149*cos(2*($F + $D + $Om) - $L);
$s += (-51613 - 42*$T)*sin($L + 2*$F + $Om) + 129*cos($L + 2*$F + $Om);
$s += (45893 + 50*$T)*sin(2*($F - $L) + $Om) + 31*cos(2*($F - $L) + $Om);
$s += (63384 + 11*$T)*sin(2*$D) - 150*cos(2*$D);
$s += (-38571 - $T)*sin(2*($F + $D + $Om)) + 158*cos(2*($F + $D + $Om));
$s += 32481*sin(2*($F - $Lp - $D + $Om));
$s -= 47722*sin(2*($D - $L)) + 18*cos(2*($D - $L));
$s += (-31046 - $T)*sin(2*($L + $F + $Om)) + 131*cos(2*($L + $F + $Om));
$s += 28593*sin($L + 2*($F - $D + $Om)) - cos($L + 2*($F - $D + $Om));
$s += (20441 + 21*$T)*sin(2*$F - $L + $Om) + 10*cos(2*$F - $L + $Om);
$s += 29243*sin(2*$L) - 74*cos(2*$L);
$s += 25887*sin(2*$F) - 66*cos(2*$F);
$s += (-14053 - 25*$T)*sin($Lp + $Om) + 79*cos($Lp + $Om);
$s += (15164 + 10*$T)*sin(2*$D - $L + $Om) + 11*cos(2*$D - $L + $Om);
$s += (-15794 + 72*$T)*sin(2*($Lp + $F - $D + $Om)) - 16*cos(2*($Lp + $F - $D + $Om));
$s += 21783*sin(2*($D - $F)) + 13*cos(2*($D - $F));
$s += (-12873 - 10*$T)*sin($L - 2*$D + $Om) - 37*cos($L - 2*$D + $Om);
$s += (-12654 + 11*$T)*sin($Om - $Lp) + 63*cos($Om - $Lp);
$s -= 10204*sin(2*($F + $D) - $L + $Om) - 25*cos(2*($F + $D) - $L + $Om);
$s += (16707 - 85*$T)*sin(2*$Lp) - 10*cos(2*$Lp);
$s -= 7691*sin($L + 2*($F + $D + $Om)) - 44*cos($L + 2*($F + $D + $Om));
$s -= 11024*sin(-2*$L + 2*$F) + 14*cos(2*($F - $L));
$s += (7566 - 21*$T)*sin($Lp + 2*($F + $Om)) - 11*cos($Lp + 2*($F + $Om));
$s += (-6637 - 11*$T)*sin(2*($F + $D) + $Om) + 25*cos(2*($F + $D) + $Om);
$s += (-7141 + 21*$T)*sin(2*($F + $Om) - $Lp) + 8*cos(2*($F + $Om) - $Lp);
$s += (-6302 - 11*$T)*sin(2*$D + $Om) + 2*cos(2*$D + $Om);
$s += (5800 + 10*$T)*sin($L + 2*($F - $D) + $Om) + 2*cos($L + 2*($F - $D) + $Om);
$s += 6443*sin(2*($L + $F - $D + $Om)) - 7*cos(2*($L + $F - $D + $Om));
$s += (-5774 - 11*$T)*sin(2*($D - $L) + $Om) - 15*cos(2*($D - $L) + $Om);
$s -= 5350*sin(2*($L + $F) + $Om) - 21*cos(2*($L + $F) + $Om);
$s += (-4752 - 11*$T)*sin(2*($F - $D) - $Lp + $Om) - 3*cos(2*($F - $D) - $Lp + $Om);
$s += (-4940 - 11*$T)*sin($Om - 2*$D) - 21*cos($Om - 2*$D);
$s += 7350*sin(2*$D - $L - $Lp) - 8*cos(2*$D - $L - $Lp);
$s += 4065*sin(2*($L - $D) + $Om) + 6*cos(2*($L - $D) + $Om);
$s += 6579*sin($L + 2*$D) - 24*cos($L + 2*$D);
$s += 3579*sin($Lp + 2*($F - $D) + $Om) + 5*cos($Lp + 2*($F - $D) + $Om);
$s += 4725*sin($L - $Lp) - 6*cos($L - $Lp);
$s -= 3075*sin(2*($F - $L  + $Om)) + 2*cos(2*($F - $L  + $Om));
$s -= 2904*sin(3*$L + 2*($F + $Om)) - 15*cos(3*$L + 2*($F + $Om));
$s += 4348*sin(2*$D - $Lp) - 10*cos(2*$D - $Lp);
$s -= 2878*sin($L - $Lp + 2*($F + $Om)) - 8*cos($L - $Lp + 2*($F + $Om));
$s -= 4230*sin($D) - 5*cos($D);
$s -= 2819*sin(2*($F + $D + $Om) - $L - $Lp) - 7*cos(2*($F + $D + $Om) - $L - $Lp);
$s -= 4056*sin(2*$F - $L) - 5*cos(2*$F - $L);
$s -= 2647*sin(2*($F + $D + $Om) - $Lp) - 11*cos(2*($F + $D + $Om) - $Lp);
$s -= 2294*sin($Om - 2*$L) + 10*cos($Om - 2*$L);
$s += 2481*sin($L + $Lp + 2*($F + $Om)) - 7*cos($L + $Lp + 2*($F + $Om));
$s += 2179*sin(2*$L + $Om) - 2*cos(2*$L + $Om);
$s += 3276*sin($Lp - $L + $D) + cos($Lp - $L + $D);
$s -= 3389*sin($L + $Lp) - 5*cos($L + $Lp);
$s += 3339*sin($L + 2*$F) - 13*cos($L + 2*$F);
$s -= 1987*sin(2*($F - $D) - $L + $Om) + 6*cos(2*($F - $D) - $L + $Om);
$s -= 1981*sin($L + 2*$Om);
$s += 4026*sin($D - $L) - 353*cos($D - $L);
$s += 1660*sin($D + 2*($F + $Om)) - 5*cos($D + 2*($F + $Om));
$s -= 1521*sin(2*($F + 2*$D + $Om) - $L) - 9*cos(2*($F + 2*$D + $Om) - $L);
$s += 1314*sin($Lp - $L + $D + $Om);
$s -= 1283*sin(2*($F - $Lp - $D) + $Om);
$s -= 1331*sin($L + 2*($F + $D) + $Om) - 8*cos($L + 2*($F + $D) + $Om);
$s += 1383*sin(2*($F - $L + $D + $Om)) - 2*cos(2*($F - $L + $D + $Om));
$s += 1405*sin(2*$Om - $L) + 4*cos(2*$Om - $L);
$s += 1290*sin($L + $Lp + 2*($F - $D + $Om));

// ----------------------------------------
// Return nutation in longitude in degrees.

   return $s / 36000000000;

} // End of  dPsi_2000B()





// ========================================================================
// Alternate trig functions for degree arguments instead of radians.
// Same names as original functions except with trailing _d chars.
//
// Arguments can be given in decimal or DMS string form.
// Example: 22.5°  =  "22 30"
//
// atan2_d() function returns longitudinal angles from 0 to +360 degrees.

   function sin_d($AngDMS)
  {
   return sin(deg2rad(DMS_to_Deg($AngDMS)));
  }

   function cos_d($AngDMS)
  {
   return cos(deg2rad(DMS_to_Deg($AngDMS)));
  }

   function tan_d($AngDMS)
  {
   return tan(deg2rad(DMS_to_Deg($AngDMS)));
  }

   function asin_d($x)
  {
   return rad2deg(asin(floatval($x)));
  }

   function acos_d($x)
  {
   return rad2deg(acos(floatval($x)));
  }

   function atan_d($x)
  {
   return rad2deg(atan(floatval($x)));
  }

   function atan2_d($y, $x)
  {
   $w = rad2deg(atan2(floatval($y), floatval($x)));

   return $w + (($w < 0)? 360:0); // 0 to +360 degrees
  }










/*
   ========================================================================
   Convert decimal degrees to degrees, minutes, seconds of arc.
   The angular elements are returned in a space-delimited string.

   Optional symbols are attached to the angle elements. The seconds can
   optionally be rounded to up to 3 decimals with 0=Default.

   ========================================================================
*/

   function Deg_to_DMS ($AngDeg, $ssDecimals=0, $PosSignFlag=FALSE, $SymbolsFlag=TRUE)
{

// Init
   $_d_ = $_m_ = $_s_ = '';

// ------------------------------------------------------
// Compute signed angular elements from decimal argument.
// This value is taken as decimal degrees.
   $sign = ($AngDeg < 0)? '-' : '';

   if ($PosSignFlag === TRUE and $sign == '') {$sign = '+';}

   $degrees = abs($AngDeg);
   $dd      = floor($degrees);
   $minutes = 60*($degrees - $dd);
   $mm      = floor($minutes);
   $seconds = 60*($minutes - $mm);
   $ss      = SPrintF("%1.3f", $seconds);

// ----------------------------------
// Check for that blasted 60s glitch.
   if ($ss == 60) {$ss = 0; $mm++;}
   if ($mm == 60) {$mm = 0; $dd++;}

// ----------------------------
// Format the angular elements.
   $dd = SPrintF("%02d",  $dd);
   $mm = SPrintF("%02d",  $mm);
   $ss = SPrintF("%1.$ssDecimals" . "f", $ss);

   if ($ss < 10) {$ss = "0$ss";}

// Attach optional DMS symbols.
// Default = TRUE.
   if ($SymbolsFlag === TRUE)
      {
       $_d_ = "&deg;";
       $_m_ = "'";
       $_s_ = '"';
      }

// Done.
   return "$sign$dd$_d_ $mm$_m_ $ss$_s_";

} // End of  Deg_to_DMS(...)


/*
   ===========================================================================
   Convert hour angle in decimal degrees into hrs, min, sec of RA.
   The angular elements are returned as an HMS or H:M:S string.

   Optional HMS symbols are attached to or colons inserted between the
   elements.  The seconds part can be rounded to up to 3 decimals (0=Default).

   ===========================================================================
*/

   function Deg_to_HMS ($AngDeg, $ssDecimals=0, $PosSignFlag=FALSE, $SymbolsFlag=TRUE)
{

// Init
   $_h_ = $_m_ = $_s_ = '';

// ---------------------------------------------------------
// Compute hour angle elements from signed decimal argument.
// This value is taken as decimal degrees.
   $sign = ($AngDeg < 0)? '-' : '';

   if ($PosSignFlag === TRUE and $sign == '') {$sign = '+';}

   $hours   = abs(trim($AngDeg) / 15);
   $hh      = floor($hours);
   $minutes = 60*($hours - $hh);
   $mm      = floor($minutes);
   $seconds = 60*($minutes - $mm);
   $ss      = SPrintF("%1.3f", $seconds);

// ----------------------------------
// Check for that blasted 60s glitch.
   if ($ss == 60) {$ss = 0; $mm++;}
   if ($mm == 60) {$mm = 0; $hh++;}

// --------------------------------------
// Format and return the angular elements
// in a space-delimited string.
   $hh = SPrintF("%02d",  $hh);
   $mm = SPrintF("%02d",  $mm);
   $ss = SPrintF("%1.$ssDecimals" . "f", $ss);

   if ($ss < 10) {$ss = "0$ss";}

// Attach optional DMS symbols.
// Default = TRUE.
   if ($SymbolsFlag === TRUE)
      {
       $_h_ = 'h';
       $_m_ = "m";
       $_s_ = 's';
      }

// Done.
   return "$sign$hh$_h_ $mm$_m_ $ss$_s_";

} // End of  Deg_to_HMS(...)


// *************************************************************************************

/*

   This function converts decimal hours into
   an hours, minutes and seconds string.

   For example, the arguments (10.123456789, 3)
   hours would return the string "10h 07m 24.444s"

   $SymbolsMode = 0, 1 or 2
                  0 = No symbols
                  1 = Use colons (:) instead of h,m,s symbols
                  2 = Use h,m,s symbols instead of colons (= Default).

*/
   function Hours_to_HMS ($hours, $ssDecimals=0, $PosSignFlag=FALSE, $SymbolsMode='h')
{

// ---------------------------
// Initialize symbol carriers.

   $_h_ = $_m_ = $_s_ = '';

// -------------------------------------------------
// Compute time elements from signed hours argument.
// Store and remember sign and work with abs. value.

   $sign = ($hours < 0)? '-' : '';

   if ($PosSignFlag === TRUE and $sign == '') {$sign = '+';}

   $hh      = floor(FloatVal($hours));
   $minutes = 60*(FloatVal($hours) - $hh);
   $mm      = floor($minutes);
   $seconds = 60*($minutes - $mm);
   $ss      = SPrintF("%1.3f", $seconds);

// ----------------------------------
// Patch for that blasted 60s glitch.

   if ($ss == 60) {$ss = 0; $mm++;}
   if ($mm == 60) {$mm = 0; $hh++;}

// --------------------------------------
// Format and return the angular elements
// in a space-delimited string.

   $hh = SPrintF("%02d",  $hh);
   $mm = SPrintF("%02d",  $mm);
   $ss = SPrintF("%1.$ssDecimals" . "f", $ss);

   if ($ss < 10) {$ss = "0$ss";}

// ----------------------------------------
// Attach optional h,m,s symbols or colons.
// Default = 'h'.

   if ($SymbolsMode == 0 or $SymbolsMode == '')
      {
       $_h_ = ' ';
       $_m_ = ' ';
       $_s_ = ' ';
      }

   if ($SymbolsMode == 1 or $SymbolsMode == ':')
      {
       $_h_ = ':';
       $_m_ = ':';
       $_s_ =  '';
      }

   if ($SymbolsMode == 2 or strtolower($SymbolsMode) == 'h')
      {
       $_h_ = 'h ';
       $_m_ = 'm ';
       $_s_ = 's';
      }


// Done.
   return "$sign$hh$_h_$mm$_m_$ss$_s_";

} // End of  Hours_to_HMS(...)






/*
   This function takes a DMS string and returns the corresponding
   value in decimals degrees.  The DMS elements are delimited by
   spaces.

   Example: Given DMS string = "9 12 50.306"
            Returned = 9.2139738888888889

*/

   function DMS_to_Deg ($DMSstr)
{

// ----------------------------------------------------------
// Read angle/time string argument and force to lower case in
// the event any (deg d h m s : ° ' ") symbols are used.

   $dms = strtolower(trim($DMSstr));

// ------------------------------------------------------
// Convert any (deg d m s ° ' ") symbols into spaces.

   $dms = str_replace('deg', ' ', $dms);
   $dms = str_replace('d',   ' ', $dms);
   $dms = str_replace('m',   ' ', $dms);
   $dms = str_replace('s',   ' ', $dms);
   $dms = str_replace('°',   ' ', $dms);
   $dms = str_replace("'",   ' ', $dms);
   $dms = str_replace('"',   ' ', $dms);

// ----------------------------------------------------
// Normalize the spacing and then split and extract the
// individual angle/time string elements (dh or hh, mm,ss).

   $dms = preg_replace("/\s+/", " ", trim($dms));
   $wdms = preg_split("[ ]", $dms);
   $wdmsCount = count($wdms);
   $dd = ($wdmsCount >= 1)? bcadd($wdms[0],"0", 20) : "0";
   $mm = ($wdmsCount >= 2)? bcadd($wdms[1],"0", 20) : "0";
   $ss = ($wdmsCount >= 3)? bcadd($wdms[2],"0", 20) : "0";

// ------------------------------------------------
// Remember and then remove any numerical (±) sign.

   $NumSign = (substr($DMSstr,0,1) == '-')? '-' : '';
   $dd = str_replace('-', '', $dd);
   $dd = str_replace('+', '', $dd);

// -------------------------------------------------------------
// If original angle argument began with a + sign, then preserve
// it so that all returned positive results will have a + sign.
// Otherwise, positive results will NOT have a + sign.

   if (substr($DMSstr,0,1) == '+') {$NumSign = '+';}

// ----------------------------------------------
// Compute decimal degrees/hours value equivalent
// to the given dhms argument elements.

   $w2 = bcadd(bcadd(bcmul($dd,"3600",20),bcmul($mm,"60",20),20),$ss,20);

// -----------------------------------------------------------
// If result equates to zero, then suppress any numerical sign.

   if (bccomp($w2, "0", 20) == 0) {$NumSign = '';}

// ---------------------------------------------------------
// Round off result to 16 decimals, recalling original sign.

   return $NumSign . bcadd(bcdiv($w2,"3600",20), "0.00000000000000005",16);

} // End of  DMS_to_Deg (...)





/*
   This function takes an HMS string and returns the corresponding
   value in decimals degrees.  The HMS elements are delimited by
   spaces.

   Example: Given HMS string = "10 11 12.246"
            Returned = 9.2139738888888889

*/

   function HMS_to_Deg ($hhmmss)
{

// -----------------------------------------------------------
// Read hour angle string argument and force to lower case in
// the event any (h m s :) symbols are used.

   $hms = strtolower(trim($hhmmss));

// ------------------------------------------------------
// Convert any (h m s :) symbols into spaces.

   $hms = str_replace('h', ' ', $hms);
   $hms = str_replace('m', ' ', $hms);
   $hms = str_replace('s', ' ', $hms);
   $hms = str_replace(':', ' ', $hms);


// ----------------------------------------------------
// Normalize the spacing and then split and extract the
// individual time string elements (hh, mm,ss).

   $hms = preg_replace("/\s+/", " ", trim($hms));
   $whms = preg_split("[ ]", $hms);
   $whmsCount = count($whms);
   $hh = ($whmsCount >= 1)? bcadd($whms[0],"0", 20) : "0";
   $mm = ($whmsCount >= 2)? bcadd($whms[1],"0", 20) : "0";
   $ss = ($whmsCount >= 3)? bcadd($whms[2],"0", 20) : "0";

// ------------------------------------------------
// Remember and then remove any numerical (±) sign.

   $NumSign = (substr($hhmmss,0,1) == '-')? '-' : '';
   $hh = str_replace('-', '', $hh);
   $hh = str_replace('+', '', $hh);

// -------------------------------------------------------------
// If original angle argument began with a + sign, then preserve
// it so that all returned positive results will have a + sign.
// Otherwise, positive results will NOT have a + sign.

   if (substr($hhmmss,0,1) == '+') {$NumSign = '+';}

// --------------------------------------
// Compute decimal hours value equivalent
// to the given hms argument elements.

   $w2 = bcadd(bcadd(bcmul($hh,"3600",20),bcmul($mm,"60",20),20),$ss,20);

// ---------------------------------
// Convert hours value into degrees.

   $w2 = bcmul($w2, '15', 20);

// -----------------------------------------------------------
// If result equates to zero, then suppress any numerical sign.

   if (bccomp($w2, "0", 20) == 0) {$NumSign = '';}

// ---------------------------------------------------------
// Round off result to 16 decimals, recalling original sign.

   return $NumSign . bcadd(bcdiv($w2,"3600",20), "0.00000000000000005",16);

} // End of  HMS_to_Deg(...)





/*
   ========================================================================
   This function performs Lagrange interpolation within an XY data table.

   ========================================================================
*/

   function Lagrange_Interp ($XvsYDataTable, $xArg)
{

// -----------
// Initialize.

   $XDataStr = $YDataStr = '';

// --------------------------------------------------------------
// Read and split XY data pairs into a work array.  In the array,
// even number indices = X-data, odd number indices = Y-data.

   $XY = preg_split("[ ]", preg_replace("/\s+/", ' ', trim($XvsYDataTable)));

// ---------------------------------------------
// Count the total number of data elements. This
// value should always be an even number.

   $TotalDataCount = count($XY);

// ------------------------------------------------------------
// Number of data pairs.  This value should be an integer value
// exactly equal to 1/2 the total number of data points. If not,
// then there is an odd mismatched data point.

   $n = $TotalDataCount / 2;

// -----------------------------------------------------------
// Return error message if data vector element count mismatch.
// For every X value there must be a corresponding Y value or
// an XY data count mismatch error occurs.  An error will also
// occur if insufficient data.  There must be at least two XY
// data points given.

   if ($TotalDataCount < 4 )
      {return "ERROR: There must be at least two XY data pairs.";}

   if ($n != floor($n + 0.5))
      {return "ERROR: XY Data Count Mismatch. Odd data element.";}

// ------------------------------------------------------------
// Compute number of XY data pairs.  This value is exactly half
// the value of the total number of data elements.

   $n = $TotalDataCount / 2;

// -------------------------------------------------------
// Construct separate XY data strings from the array data.
// The XY data strings should each contain the same number
// of data elements.

   for($i=0;   $i < $TotalDataCount;   $i+=2)
      {
       $XDataStr .= $XY[$i]   . " ";
       $YDataStr .= $XY[$i+1] . " ";
      }

// --------------------------------------------------------------
// Split the created XY data vector strings into matching indexed
// arrays.  For every X value there must be a matching Y value
// and no two X values can be identical.

   $X = preg_split("[ ]", trim($XDataStr));
   $Y = preg_split("[ ]", trim($YDataStr));

// ----------------------------------------
// Read X argument for which to interpolate
// the Y value from the given XY data.

   $x = trim($xArg);  if ($x == "") {$x = "0";}

// -----------------------------------
// Initialize Y summation accumulator.
   $y = 0.0;

// -----------------------------------------------------
// Compute LaGrangian product (Li) for given X argument.

   for ($i=0;   $i < $n;   $i++)
       {
        $Li = 1.0;

        for ($j=0;   $j < $n;   $j++)
            {
             if ($j != $i) // Skip this cycle when j == i
                {
                 $Li = ($Li * ($x - $X[$j])) / ($X[$i] - $X[$j]);
                }
            } // Next j

//      --------------------------------------
//      Accumulate sum of Yi polynomial terms.

        $y += ($Y[$i] * $Li);

       } // Next i

   return $y;

} // End of  Lagrange_Interp(...)




/*
   ========================================================================
   This function simply swaps all matching XY data pairs held in a
   space delimited string argument. It was designed as a companion
   to the Lagrange_Interpolate() function to easily facilitate the
   swapping of the XY data computation roles, but it can be used
   for any 2-column data table.

   ERRORS:
   On error FALSE is returned.  An error occurs if
   odd number of data elements or < 4 elements.
   ========================================================================
*/

   function Swap_Data_Columns ($XvsYDataTable)
{

// -------------------------------------------------------------------------
// Read space-delimited data string and split items into a sequential array
// where even indices represent the X-data and odd represents the Y-data.

   $XYData = trim($XvsYDataTable);

// ------------------------------------------------------
// Replace any comma or semicolon delimiters with spaces.

   $XYData = str_replace(";", " ", $XYData);
   $XYData = str_replace(",", " ", $XYData);

// -------------------------------------------------------------------------
// Read space-delimited data string and split items into a sequential array
// where even indices represent the X-data and odd represents the Y-data.

   $XYArray = preg_split("[ ]", preg_replace("/\s+/", " ", trim($XYData)));

   $XYArrayCount = count($XYArray);

   if ($XYArrayCount % 2 != 0 or $XYArrayCount < 4) {return FALSE;}

   $XYPairsCount = $XYArrayCount / 2;

// ------------------------------------------------------
// Construct and return output table of swapped XY pairs.

   $SwappedYvsXDataTable = "";

   for ($i=1;   $i < $XYArrayCount;   $i+=2)
       {
        $SwappedYvsXDataTable .= $XYArray[$i] . " " . $XYArray[$i-1] . "\n";
       }

   return trim($SwappedYvsXDataTable);
}








/*
  ----------------------------------------------------------------------
  This function returns the time of an extremum (minimum or maximum), if
  any, within a given time vs event table.  For example, this function
  can be used to find the times of perihelion, aphelion, perigee, apogee
  or any general periapsis or apoapsis times.  It is based on a 5-point
  data table and the extremum is computed from a polynomial derived from
  the given data.

  ERRORS:
  No error checking is done and the function assumes that an extremum
  exists within the given data table.
  ----------------------------------------------------------------------
*/


   function Extremum5 ($DataTableStr)
{

// ---------------------------------
// Read data table and parse values.
   $DataTable = preg_replace("/\s+/", ' ', trim($DataTableStr));

   list ($x1,$y1, $x2,$y2, $x3,$y3, $x4,$y4, $x5,$y5)
   = preg_split("[ ]", $DataTable);

// Set constant interval between points.
   $interval = $x2 - $x1;

// Compute systematic differentials
   $a = $y2 - $y1;
   $b = $y3 - $y2;
   $c = $y4 - $y3;
   $d = $y5 - $y4;

   $e = $b - $a;
   $f = $c - $b;
   $g = $d - $c;
   $h = $f - $e;
   $i = $g - $f;
   $j = $i - $h;

   $k = $j/24;
   $m = ($h + $i)/12;
   $n = $f/2 - $k;
   $p = ($b + $c)/2 - $m;

   $q = $r = 0;

// Perform extremum zeroing loop for no more than 25 cycles.
   while ($r < 25)
  {
   $s = 6*($b + $c) - $h - $i + 3*$q*$q*($h + $i) + 2*$q*$q*$q*$j;
   $t = $j - 12*$f;
   $q = $s/$t;

   $r++;
  }
   return $q * $interval + $x3; // Done.

} // End of  Extremum5(...)





/*
  ========================================================================
  Level 0

  This function computes the theoretical astronomical delta-T estimate
  in seconds for any dates between -2000 and 3000, based on the NASA
  polymomial expressions for Delta T

  NASA Ref:
  http://eclipse.gsfc.nasa.gov/LEcat5/deltatpoly.html


  ARGUMENTS:
  Y = Year number (-2000 BC to 3000 AD) (Neg=BC)(no year zero)
  mNum = Month number(01 to 12)
  EXAMPLE: Ym = 195005 (= 1950 May)

  ERRORS:
  FALSE is returned on error.  An error results if either
  the year or month number is invalid. A zero year value
  will return an error.
  ========================================================================
*/

   function Delta_T($Ym)
{

// ----------------------------------------------
// ERROR if year value is non-numeric. It must be
// a number in the range from -2000 to +3000.
// Ym = 100*Y + m
   if (!is_numeric(substr($Ym,0,6))) {return FALSE;}


// Parse Ym value.
   $Y = substr($Ym,0,4);
   $mNum = intval(Ltrim(substr($Ym,4,2), '0'));

// ------------------------------------------
// ERROR if year value is invalid. It must be
// in the range from -2000 to +3000.
   if ($Y < -2000 or $Y == 0 or $Y > 3000) {return FALSE;}

// -------------------------------------------------------
// ERROR if month value is outside the range from 1 to 12.
   if ($mNum < 1 or $mNum > 12) {return FALSE;}

// -----------------------------------------------------------
// If a negative (BC) year is given , it is converted into the
// corresponding mathematical year by adding 1.
//
// The year 100 BC (-100), becomes the mathematical year -99.

   $y = $Y + (($Y < 0)? 1:0) + ($mNum - 0.5)/12;


   if ($Y >= -2000 and $Y < -500)
      {
       $u  = ($Y - 1820) / 100;

       return 32*$u*$u - 20;
      }


   if ($Y >= -500 and $Y < 500)
      {
       $u  = $y/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 10583.6 - 1014.41*$u
                      +   33.78311*$u2
                      -    5.952053*$u3
                      -    0.1798452*$u4
                      +    0.022174192*$u5
                      +    0.0090316521*$u6;
      }


   if ($Y >= 500 and $Y < 1600)
      {
       $u  = ($y - 1000)/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 1574.2 - 556.01*$u
                     +  71.23472*$u2
                     +   0.319781*$u3
                     -   0.8503463*$u4
                     -   0.005050998*$u5
                     +   0.0083572073*$u6;
      }


   if ($Y >= 1600 and $Y < 1700)
      {
       $u  = $y - 1600;

       return 120 - 0.9808*$u - 0.01532*$u*$u + $u*$u*$u/7129;
      }


   if ($Y >= 1700 and $Y < 1800)
      {
       $u  = $y - 1700;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 8.83 + 0.1603*$u
                   - 0.0059285*$u2
                   + 0.00013336*$u3
                   - $u4/1174000;
      }


   if ($Y >= 1800 and $Y < 1860)
      {
       $u  = $y - 1800;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;
       $u7 = $u * $u6;

       return 13.72 - 0.332447*$u
                    + 0.0068612*$u2
                    + 0.0041116*$u3
                    - 0.00037436*$u4
                    + 0.0000121272*$u5
                    - 0.0000001699*$u6
                    + 0.000000000875*$u7;
      }


   if ($Y >= 1860 and $Y < 1900)
      {
       $u  = $y - 1860;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 7.62 + 0.5737*$u
                   - 0.251754*$u2
                   + 0.01680668*$u3
                   - 0.0004473624*$u4
                   + $u5 / 233174;
      }


   if ($Y >= 1900 and $Y < 1920)
      {
       $u  = $y - 1900;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 1.494119*$u - 2.79
                          - 0.0598939*$u2
                          + 0.0061966*$u3
                          - 0.000197*$u4;
      }


   if ($Y >= 1920 and $Y < 1941)
      {
       $u  = $y - 1920;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 21.20 + 0.84493*$u - 0.076100*$u2 + 0.0020936*$u3;
      }


   if ($Y >= 1941 and $Y < 1961)
      {
       $u  = $y - 1950;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 29.07 + 0.407*$u - $u2/233 + $u3/2547;
      }


   if ($Y >= 1961 and $Y < 1986)
      {
       $u = $y - 1975;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 45.45 + 1.067*$u - $u2/260 - $u3/718;
      }


   if ($Y >= 1986 and $Y < 2005)
      {
       $u  = $y - 2000;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 63.86 + 0.3345*$u
                    - 0.060374*$u2
                    + 0.0017275*$u3
                    + 0.000651814*$u4
                    + 0.00002373599*$u5;
      }


   if ($Y >= 2005 and $Y < 2050)
      {
       $u  = $y - 2000;

       return 62.92 + 0.32217*$u + 0.005589*$u*$u;
      }


   if ($Y >= 2050 and $Y <= 2150)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 0.5628*(2150 - $y) - 20;
      }


   if ($Y > 2150 and $Y <= 3000)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 20;
      }

   return FALSE;

} // End of  Delta_T(...)













/*
   ========================================================================
   LEVEL-1 FUNCTIONS:

   FUNCTIONS BELOW ARE LEVEL-1 FUNCTIONS, MEANING THAT THEY CALL AT LEAST 1
   OR MORE LEVEL-0 FUNCTIONS ABOVE.  THEY CANNOT BE TRANSPORTED WITHOUT
   ALSO INCLUDING THE LEVEL-0 FUNCTIONS THEY DEPEND ON AS WELL.
   ========================================================================
*/




/*
   Ths is a special version of the Delta_T(...) function.  It will return
   the Delta T string in ±HH:MM:SS format, resolved to the nearest second.

   NOTE:
   This function requires:
   Delta_T(...)
   Hours_to_HMS(...)

   No error checks are internally performed.

   If given a full date value (Ymd), it will only use the (Ym) parts.

*/

   function Delta_T_HMS ($Ym)
{
   return Hours_to_HMS(round(Delta_T($Ym), 0)/3600, 0, TRUE, ":");
}















/*
   This function computes the nutation in right ascension or the equation
   of the equinoxes, in degrees.

   This value is the difference (True Equinox) minus (Mean Equinox).

   The IAU 2000B nutation series is applied.

   LEVEL 0 DEPENDENCIES:
   Mean_Eps()
   dEps()
   dPsi()


   NOTE:
   Degrees * 240 = Seconds of RA
   ========================================================================
*/

   function dEquinox_2000B($JDTT)
{

// ---------------------------------------------
// Compute cosine of true obliquity of ecliptic.

   $CosEpsTrue = cos(deg2rad(Mean_Eps($JDTT) + dEps_2000B($JDTT)));

// --------------------------------------------
// Return equation of the equinoxes in degrees.

   return dPsi_2000B($JDTT) * $CosEpsTrue;

} // End of  dEquinox()







/*
   This function takes an HMS string and returns the corresponding
   value in decimals hours.  The HMS elements are delimited by
   spaces.

   Example: Given HMS string = "10 11 12.246"
            Returned = 10.1867350000000000 deg

*/

   function HMS_to_Hours ($hhmmss)
{

// -----------------------------------------------------------
// Read hour angle string argument and force to lower case in
// the event any (h m s :) symbols are used.

   $hms = strtolower(trim($hhmmss));

// ------------------------------------------------------
// Convert any (h m s :) symbols into spaces.

   $hms = str_replace('h', ' ', $hms);
   $hms = str_replace('m', ' ', $hms);
   $hms = str_replace('s', ' ', $hms);
   $hms = str_replace(':', ' ', $hms);


// ----------------------------------------------------
// Normalize the spacing and then split and extract the
// individual time string elements (hh, mm,ss).

   $hms = preg_replace("/\s+/", " ", trim($hms));
   $whms = preg_split("[ ]", $hms);
   $whmsCount = count($whms);
   $hh = ($whmsCount >= 1)? bcadd($whms[0],"0", 20) : "0";
   $mm = ($whmsCount >= 2)? bcadd($whms[1],"0", 20) : "0";
   $ss = ($whmsCount >= 3)? bcadd($whms[2],"0", 20) : "0";

// ------------------------------------------------
// Remember and then remove any numerical (±) sign.

   $NumSign = (substr($hhmmss,0,1) == '-')? '-' : '';
   $hh = str_replace('-', '', $hh);
   $hh = str_replace('+', '', $hh);

// -------------------------------------------------------------
// If original angle argument began with a + sign, then preserve
// it so that all returned positive results will have a + sign.
// Otherwise, positive results will NOT have a + sign.

   if (substr($hhmmss,0,1) == '+') {$NumSign = '+';}

// --------------------------------------
// Compute decimal hours value equivalent
// to the combined hh,mm,ss values.

   $w2 = bcadd(bcadd(bcmul($hh,"3600",20),bcmul($mm,"60",20),20),$ss,20);

// ------------------------------------------------------------
// If result equates to zero, then suppress any numerical sign.

   if (bccomp($w2, "0", 20) == 0) {$NumSign = '';}

// ---------------------------------------------------------
// Round off result to 16 decimals, recalling original sign.

   $w =  bcadd(bcdiv($w2,3600, 20), "0.00000000000000005", 16);

// Trim off any redundant zeros/decimal point.
   return $NumSign . rtrim(rtrim($w, '0'), '.');

} // End of  HMS_to_Hours(...)



/*
   This function adjusts the right-column values
   of an interpolation table of degree or hour
   angles to account for the 24/0 or 360/0
   cyclic transition points.

   mode = 24 for hour angles expressed in hours.
        = 360 for angles expressed in degrees.
*/

   function xxx_24_360 ($AngleTableString, $mode=360)
{
// Convert (\n) codes into spaces and then remove any redundant white space.
   $AngTableStr = preg_replace("/\s\s+/", " ", trim(preg_replace("[\n]"," ", $AngleTableString)));

// Put paired space-delimited elements into array.
   $wArray = preg_split("[ ]", $AngTableStr);

// Count total array elements.  Should always be an even
// number, since it equates to two times the number of
// data pairs.  The odd indexed elements are the angle
// values and the even indexed elements are the JD12TT
// values.
   $wArrayCount = count($wArray);

// Get first and final angle values from array.
   $FirstAng = substr($wArray[1], -9);
   $FinalAng = substr($wArray[$wArrayCount-1], -9);

// Cycle through array, adding adjustment to all the
// angles that have values < first angle value.
   $AngTableStr = "";
   for($i=1;   $i <= $wArrayCount;   $i+=2)
      {
       $wArray[$i]  += (($wArray[$i] < $FirstAng)? $mode:0);
       $AngTableStr .= ($wArray[$i-1] . " " . $wArray[$i] . "\n");
      }

// Done.
   return trim($AngTableStr);
}





/*
   This function computes the local azimuth (direction) and
   altitude relative to the local horizon, in degrees.

   NOTE: This function depends on globals LngDeg, LatDeg

   JDTT    = Astronomical JD argument corresponding to date/time
   RAHrs   = Right Ascension of body, in HOURS - not degrees.
   DeclDeg = Declination of body, from 0 to ±90 degrees.

   Output  = Horizon coordinates string = "AzimDeg DeclDeg"
   AzimDeg = Azimuth (Direction) from 0 to 360° (North = 0°/360°).
   AltDeg  = Altitude relative to horizon (Neg = Below horizon).
*/

   function Azim_Alt ($JDTT, $RAHrs, $DeclDeg)
{
// Get external global geographic location variables.
   global $LngDeg, $LatDeg;

// Compute local true sidereal time.
   $LSTTrueDeg = Mean_Sid_Time($JDTT, $LngDeg) + dEquinox_2000B($JDTT);

// Compute local hour angle of body.
   $H = $LSTTrueDeg - $RAHrs*15;

// Compute azimuth clockwise from 0 to 360 degrees (North = 0 degrees).
   $AzimDeg = 180 + atan2_d(sin_d($H), (cos_d($H)*sin_d($LatDeg) - tan_d($DeclDeg)*cos_d($LatDeg)));
   $AzimDeg -= (($AzimDeg >= 360)? 360:0);

// Compute altitude relative to local horizon.  Negative = Below horizon.
   $SinAlt = sin_d($LatDeg)*sin_d($DeclDeg) + cos_d($LatDeg)*cos_d($DeclDeg)*cos_d($H);
   $AltDeg = asin_d($SinAlt);

   return "$AzimDeg $AltDeg";
}









/*
   ========================================================================
   This function returns the compass symbol corresponding to
   azimuth angle in degrees and zero-mode.  The default zero
   azimuth direction is north.

   ZeroModeSymbol = 'N' (North = 0 convention = Default)
   ZeroModeSymbol = 'S' (South = 0 convention = Optional)

   ========================================================================
*/

   function Compass_Symbol ($AzimDeg, $ZeroModeSymbol='N')
{
   $a = floatval($AzimDeg) + ((strtoupper($ZeroModeSymbol) == 'S')? 180:0);

   $a -= 360*floor($a/360);

   $i = ($a < 11.25 or $a > 348.75)? 0 : floor(0.5 + $a/22.5);

   if ($ZeroModeSymbol != 'S')
   {return trim(substr("N  NNENE ENEE  ESESE SSES  SSWSW WSWW  WNWNW NNWN", 3*$i, 3));}
   else
   {return trim(substr("S  SSWSW WSWW  WNWNW NNWN  NNENE ENEE  ESESE SSES", 3*$i, 3));}
}




   function LZPad_Number($NumberVal, $NumPaddingZeros=0)
{
   $dp = strpos($NumberVal, ".");

   if ($dP === FALSE)
      {
       return SPrintF("%0$nz", $NumberVal);
      }
   else
      {
       $i = SPrintF("%0$nz", substr($NumberVal, 0, $dp-1));
       $d = substr($NumberVal, $dp, strlen($NumberVal));
       return "$i$d";
      }
}



/*
   ===================================================
   This function computes the angular diameter of a
   body in seconds of arc given the distance in AU
   and angular diameter of the body in seconds of
   arc at 1.0 AU.  It applies to the Sun, Moon,
   Planets and Pluto.

   Zero is returned if the case-insensitive body ID
   is not found in the switch/case list.

   BodyName = Sun, Moon or planet name.  Uses only the
              first two characters in the body name as
              ID and all other characters are ignored.
              The body name is NOT case-sensitive.

   DistAU   = Geocentric distance in AUs.

   The returned angular diameter is in seconds of arc.

   NOTE:
   Angular diameter of Pluto has been changed to a
   newer value based on New Horizons probe data.
   ===================================================
*/

   function Ang_Diam_SS($BodyName, $DistAU)
{
   switch (substr(strtolower($BodyName), 0,2))
  {
   case "su" : $At1AU = 1919.26;       break;
   case "mo" : $At1AU =    4.79021184; break;
   case "ea" : $At1AU =   17.5580658;  break;
   case "me" : $At1AU =    6.72;       break;
   case "ve" : $At1AU =   16.68;       break;
   case "ma" : $At1AU =    9.36;       break;
   case "ju" : $At1AU =  196.88;       break;
   case "sa" : $At1AU =  165.46;       break;
   case "ur" : $At1AU =   70.04;       break;
   case "ne" : $At1AU =   67.00;       break;
   case "pl" : $At1AU =    3.2677;     break;
   default   : $At1AU =     0;         break;
  }

// Return angular diameter of body in seconds
// of arc, rounded to 10 decimals.
   return SPrintF("%1.10f", $At1AU/$DistAU);

}// End of  Ang_Diam_SS(...)




/*
   -------------------------------------------------------------------
   This function computes the basic geocentric statistics for the Sun.

   The argument is simply the JDTT corresponding to the date/time for
   which the basic statistics of the Sun are required.

   The computed statistics are simply the apparent geocentric position
   and true distance of the Sun returned in a space-delimited string.

   $RAHrs   = RA (Right Ascension) in hours
   $DeclDeg = Declination in degrees
   $DistAU  = True distance in AUs
   -------------------------------------------------------------------
*/
   function Geocentric_Sun_Stats ($JDTTArg)
  {
// Initialize interpolation data table strings.
   $RAHrsTableStr = "";
   $DeclDegTableStr = "";
   $DistAUTableStr = "";

// Isolate JDTT value, in case of string.
   $JDTT = floatval($JDTTArg);

// ERROR if JDTT argument is outside year range from 1600 to 2200.
   if ($JDTT < 2305447.5 or $JDTT > 2524961.5) {return FALSE;}

// Determine JD12TT value of middle table row.
   $JD12TTMiddle = floor($JDTT);

// Read sun data into working array and count total data file lines.
   $SunDataArray = file("data/geocen-equ-Sun-1600-to-2200.data");
   $SunDataCount = count($SunDataArray);

// Read each file line from the beginning and process ONLY data lines
// that begin with any digit (0 to 9).  All lines NOT beginning with
// a digit are treated as comment lines and ignored.
   for ($i=0;   $i < $SunDataCount;  $i++)
  {
// Get current Sun data line
   $SunDataLine = trim($SunDataArray[$i]);

// Process the line only if it is a numerical data line.
// Otherwise, ignore and skip it.
   if (is_numeric(substr($SunDataLine, 0,1)))
  {
// Find first JD12TT value >= JDTT argument.  This value becomes
// the middle line of a 7-point Lagrange interpolation table.
   $JD12TT = substr($SunDataLine, 0,7);

   if ($JD12TT == $JD12TTMiddle)
  {
// Construct the 7-point interpolation
// tables for each of the statistics.
   for ($j = $i-3;   $j <= $i+3;   $j++)
  {
   $SunDataLine = trim($SunDataArray[$j]);
   $JD12TT      = substr($SunDataLine, 0,7);
   $RAHrs       = SPrintF("%1.7f", substr($SunDataLine,  7, 9)/1E7);
   $DeclDeg     = SPrintF("%1.6f", substr($SunDataLine, 16, 9)/1E6);
   $DistAU      = SPrintF("%1.9f", substr($SunDataLine, 25, strlen($SunDataLine))/1E9);

// Construct and append current lines to
// their respective interpolation tables.
   $RAHrsTableStr   .= "$JD12TT $RAHrs\n";
   $DeclDegTableStr .= "$JD12TT $DeclDeg\n";
   $DistAUTableStr  .= "$JD12TT $DistAU\n";
  } // End of  for(j ...)
  } // End of  if(JD12TT ...)
  } // End of  if(is_numeric ...)
  } // End of  for(i ...)

// Adjust RA angle table for interpolation, just in case
// there is a 24/00 hour transition point within it, and
// then perform Lagrange interpolation for RA at JDTT.
   $RAHrsTableStr = xxx_24_360($RAHrsTableStr, 24);
   $RAHrs = Lagrange_Interp($RAHrsTableStr, $JDTT);
   $RAHrs -= (($RAHrs > 24)? 24:0);

// Perform interpolation for declination at JDTT.
   $DeclDeg = Lagrange_Interp($DeclDegTableStr, $JDTT);

// Perform interpolation for distance at JDTT.
   $DistAU = Lagrange_Interp($DistAUTableStr, $JDTT);

// Format computed Sun statistics for output.
   $RAHrs   = SPrintF("%1.7f",  $RAHrs);
   $DeclDeg = SPrintF("%+1.6f", $DeclDeg);
   $DistAU  = SPrintF("%1.9f",  $DistAU);

// Done.
   return "$RAHrs $DeclDeg $DistAU";

} // End of  Geocentric_Sun_Stats(...)



/*
  This function computes the local date and time
  of perihelion, when the Earth is at its least
  distance from the sun during the year, for any
  Gregorian year in the range from 1600 to 2200,
  in any standard time zone.

  The TT of the event is first computed and then
  the TT is converted into local standard time
  according to the given time zone offset.

  Both local standard time and TT of the event
  are computed.

  DEPENDENCY:
  Astronomy_101.php

*/

   function Earth_Perihelion_Stats ($Y, $TZhhmm='+00:00', $dTFlag=TRUE)
{

// Tidy up TZ string, if needed.
   $TZhhmm = substr($TZhhmm = Hours_to_HMS(HMS_to_Hours($TZhhmm), 0, TRUE, ':'), 0,6);

   $PerihelionTableString = Earth_Perihelion_Table ($Y, $TZhhmm);

// Compute perihelion JDTT value and the corresponding ISO
// date/time string for the given standard time zone offset.
   $PeriJDTT      = Extremum5($PerihelionTableString);
   $PeriYmdhhmmss = substr(JD_Zone_to_Date_Time_Zone("$PeriJDTT $TZhhmm", $dTFlag), 0,17);

// Construct perihelion date/time output string.
   $m   = substr($PeriYmdhhmmss, 4,2);
   $Mmm = trim(substr('January  February March    April    May      June     July     August   SeptemberOctober  November December', 9*($m-1), 9));
   $Mmm = substr($Mmm, 0,3);
   $d   = substr($PeriYmdhhmmss, 6,2);

   $PeriDateTime = substr($PeriYmdhhmmss, 0,4) . " $Mmm $d " . substr($PeriYmdhhmmss, -8);
   $JDDate       = intval(Ymd_HMS_TZ_to_JD(intval($PeriYmdhhmmss, FALSE) . '+00:00 12'));
   $DoWIndex     = (floor($JDDate + 0.5) + 1) % 7;
   $DoWStr       = trim(substr('Sunday   Monday   Tuesday  WednesdayThursday Friday   Saturday', 9*($DoWIndex), 9));
   $DoWStr       = substr($DoWStr, 0,3);

// Patch to round up time to nearest minute.
   $hhmmss = substr($PeriDateTime, 12, 8);
   $hhmm = round_hms($hhmmss, 'hm');
   $PeriDateTime = substr($PeriDateTime, 0, 12) . $hhmm . substr($PeriDateTime, 20, strlen($PeriDateTime));

// Done.
   return trim("$PeriDateTime $TZhhmm $DoWStr $PeriJDTT");

} // End of  Earth_Perihelion_Stats(...)





/*
   This function generates a perihelion table for any given
   Greyear in the range from 1601 to 2200
*/

   function Earth_Perihelion_Table ($Y)
{
// Compute JD12TT for Dec 20th of previous year.
// This date is where the search for aphelion begins
// and spans 31 days to Jan 20th of the given year.
   $JD12TTStart = intval(Ymd_HMS_TZ_to_JD(($Y-1) . "1220 12:00 +00:00", FALSE));

// Read solar distance data table into working array.
   $DataArray = file("data/geocen-equ-Sun-1600-to-2200.data");

// Count total lines in data array.
   $DataArrayCount = count($DataArray);

// Find data line corresponding to JD12TTStart
   for ($i=0;   $i < $DataArrayCount;   $i++)
       {
        $JD12TT = substr(trim($DataArray[$i]), 0,7);

        if ($JD12TT == $JD12TTStart) {break;}
       }

// FIND DATA LINE WHERE SEQUENTIAL DIFFERENCE IS > 0.

// ==============================================
// Start searching from Dec 20th of previous year
// to Jan 20th of given year for date where the
// numerical sign of forward differences in the
// geocentric distances changes from negative
// to positive.

   $dDistAU = $PrevDistAU = 1;

   for ($j = -1;   $j < 33;   $j++)
  {
   $SunDataLine = trim($DataArray[$i+$j]);
   $wJD12TT     = substr($SunDataLine, 0,7);
   $wDistAU     = SPrintF("%1.9f", substr($SunDataLine, -10) / 1E9);
   $dDistAU     = $wDistAU - $PrevDistAU;
   $PrevDistAU  = $wDistAU;
   $dDistAU     = SPrintF("%+1.9f",  $dDistAU);

// Check for numerical sign change and break out
// when found.  This points to the transition date
// on or near the date of perihelion.
   if($j > -1 and $dDistAU > 0) {break;}
  }

// Construct 5-point data table centered on -/+ transition date.
   $PerihelionTableString = "";
   $PrevDistAU = 2;
   $p = 0;

   for ($k = $i+$j-3;   $k <= $i+$j+2;   $k++)
  {
   $p++;
   $SunDataLine = trim($DataArray[$k]);
   $wJD12TT     = substr($SunDataLine, 0,7);
   $wDistAU     = SPrintF("%1.9f", substr($SunDataLine, -10) / 1E9);
   $dDistAU     = SPrintF("%+1.9f",  $wDistAU - $PrevDistAU);
   $PrevDistAU  = $wDistAU;

// Append line to table, if (p > 1).
   if($p > 1) {$PerihelionTableString .= "$wJD12TT $wDistAU\n";}
  }

// Done.
   return trim($PerihelionTableString);

} // End of  Earth_Perihelion_Table(...)








/*
   ===========================================================
   This function returns the basic geocentric statistics for
   the Sun, Moon, Planets, Pluto and the 15 asteroids of the
   USNO asteroid ephemeris of 1998.  Body names are NOT
   case-sensitive, but must be correctly spelled.

   ARGUMENT:
   JDTT = Astronomical JD number corresponding to time of event
          on the TT scale (TT = UT + Delta T).

   The geocentric statistics are returned in a space-delimited
   string and are:
   RAHrs = Right Ascension in hours
   DeclDeg = Declination in degrees

   DistAU  = Distance in AU (Sun and Planets)
   or
   DistKm  = Distance in kilometers (Moon)

   VMag = Ideal visual stellar magnitude

   PhaseAngDeg = Phase angle (if BodyName == Moon)
   ===========================================================
*/

   function Geocentric_Ephem_Stats ($BodyName, $JDTT)
  {

// Adjust text case if necessary.
   $BodyName = ucfirst(strtolower($BodyName));

   $VisMag = '';

// Define names of bodies (Sun,Moon,Planets,Pluto).
   $BodyNames = 'SunMoonMercuryVenusMarsJupiterSaturnUranusNeptunePluto';

// Define names of the 15 USNO AE95 asteroids.
   $AsteroidNames = 'CeresPallasJunoVestaHebeIrisFloraMetisHygieaEunomiaPsycheEuropaCybeleDavidaInteramnia';

// Error returns FALSE if body name is misspelled or not
// found in the list. (NOT case sensitive to user).
   if (strpos($AsteroidNames, $BodyName) === FALSE
   and strpos($BodyNames, $BodyName) === FALSE)
      {return FALSE;}

// Set valid range of ephemeris according to body type.
   if (strpos($AsteroidNames, $BodyName) !== FALSE)
      {$AsteroidFlag = TRUE;  $ValidRangeStr = '_1800_to_2099';}
   else
      {$AsteroidFlag = FALSE; $ValidRangeStr = '-1600-to-2200';}

// Initialize interpolation data table strings.
   $RAHrsTableStr = $DeclDegTableStr = $DistAUTableStr = '';
   $PhaseAngDegTableStr = $VisMagTableStr = '';

// Force body name start with capital letter followed by all lowercase.
   $BodyName = ucfirst(strtolower(trim($BodyName)));

// Isolate JDTT value, in case of string.
   $JD = floatval($JDTT);

// ERROR if JDTT argument is outside range from 1600 to 2200.
   if ($JD < 2305447.5 or $JD > 2524961.5) {return FALSE;}

// Determine JD12TT value of middle table row.
   $JD12TTMiddle = floor($JD);

// Read body data into working array and count total data file lines.
   $BodyDataArray = file('data/geocen-equ-' . $BodyName . $ValidRangeStr . '.data');

   $BodyDataCount = count($BodyDataArray);

// Read each file line from the beginning and process ONLY data lines
// that begin with any digit (0 to 9).  All lines NOT beginning with
// a digit are treated as comment lines and ignored.
   for ($i=0;   $i < $BodyDataCount;  $i++)
  {
// Get current text line.
   $DataLine = trim($BodyDataArray[$i]);

// Process the line only if it is a data line.
   if (is_numeric(substr($DataLine, 0,1)))
  {
// Find first JD12TT value >= JDTT argument.  This value becomes
// the middle line of a 7-point Lagrange interpolation table.
   $JD12TT = substr($DataLine, 0,7);

   if ($JD12TT == $JD12TTMiddle)
  {
// Construct the 7-point interpolation
// tables for each of the statistics.
   for ($j = $i-3;   $j <= $i+3;   $j++)
  {
   $DataLine = trim($BodyDataArray[$j]);
   $JD12TT   = substr($DataLine, 0,7);
   $RAHrs    = SPrintF('%1.7f', substr($DataLine,  7,9)/1E7);
   $DeclDeg  = SPrintF('%1.6f', substr($DataLine, 16,9)/1E6);

// Determine if body is Moon.  If so, then distance
// is returned in kilometers, otherwise distance is
// returned in AUs for all other bodies.
   $PhaseAngDeg = '';
   if ($BodyName == 'Moon')
  {
   $DistAUorKm = SPrintF('%1.3f',substr($DataLine, 25, 9)/1E3);

   $PhaseAngDeg = ' ' . SPrintF('%1.5f',substr($DataLine, 34, strlen($DataLine))/1E5);
  }
 else
  {
   $DistAUorKm = substr($DataLine, 25, 2) . '.' . substr($DataLine, 27, 9);
  }

// Get visual magnitude, if applicable.
   $VisMag = '';
   if ($BodyName != 'Sun' and $BodyName != 'Moon')
  {
  if ($BodyName == 'Sun')
   {$VisMag = SPrintF('%+1.2f', substr($DataLine, 36, strlen($DataLine))/10);}
   else
   {$VisMag = SPrintF('%+1.1f', substr($DataLine, 36, strlen($DataLine))/10);}
  }

// Construct current lines for each interpolation table.
   $RAHrsTableStr       .= "$JD12TT $RAHrs\n";
   $DeclDegTableStr     .= "$JD12TT $DeclDeg\n";
   $DistAUTableStr      .= "$JD12TT $DistAUorKm\n";
   $PhaseAngDegTableStr .= "$JD12TT $PhaseAngDeg\n";
   $VisMagTableStr      .= "$JD12TT $VisMag\n";

  } // End of  for(j ...)
  } // End of  if(JD12TT ...)
  } // End of  if(is_numeric ...)
  } // End of  for(i ...)


// Adjust RA angle table for interpolation, just in case
// there is a 24/00 hour transition point within it, and
// then perform Lagrange interpolation for RA at JDTT.
   $RAHrsTableStr = xxx_24_360($RAHrsTableStr, 24);
   $RAHrs = Lagrange_Interp($RAHrsTableStr, $JDTT);
   $RAHrs -= (($RAHrs > 24)? 24:0);

// Perform interpolation for declination at JDTT.
   $DeclDeg = Lagrange_Interp($DeclDegTableStr, $JDTT);

// Perform interpolation for distance at JDTT.
   $DistAUorKm = Lagrange_Interp($DistAUTableStr, $JDTT);

// Perform interpolation for visual magnitude.
   $VisMag = '';
   if ($BodyName != 'Sun' and $BodyName != 'Moon')
  {
   $VisMag = Lagrange_Interp($VisMagTableStr, $JDTT);

   if ($BodyName == 'Sun')
      {$VisMag = trim(SPrintF("%+1.2f", $VisMag));}
   else
      {$VisMag = trim(SPrintF("%+1.1f", $VisMag));}
  }
   $VisMag = substr("$VisMag      ", 0,6);

// Format computed statistics for output.
   $RAHrs      = SPrintF("%1.7f",  $RAHrs);
   $DeclDeg    = SPrintF("%+1.6f", $DeclDeg);
   $DistAUorKm = SPrintF("%1.9f",  $DistAUorKm);

// If body is Moon, then use 3 decimals for lunar distance
// in km and append lunar phase angle to end of output.
   if ($BodyName == 'Moon')
  {
   $DistAUorKm = SPrintF("%1.3f", $DistAUorKm);
   $PhaseAngDegTableStr = xxx_24_360($PhaseAngDegTableStr, 360);
   $PhaseAngDeg = Lagrange_Interp($PhaseAngDegTableStr, $JDTT);
   $PhaseAngDeg -= (($PhaseAngDeg > 360)? 360:0);
   $PhaseAngDeg = SPrintF("%1.5f",  $PhaseAngDeg);
  }

// Compute stellar magnitude of sun, if applicable.
   if ($BodyName == 'Sun'){$VisMag = SPrintF("%+1.2f", 5*log10($DistAUorKm) - 26.74);}

// Done.
   return trim(strip_wspace("$RAHrs $DeclDeg $DistAUorKm $PhaseAngDeg") . " " . trim($VisMag));

} // End of  Geocentric_Ephem_Stats(...)




/*
   This function generates a basic ephemeris statistics table
   for any given body for any given month and year within the
   valid span for the given body.

   The returned statistics for each date are:
   "JD12TT RAHrs DeclDeg DistAUorKm dd"

*/

   function Body_Month_Table ($BodyName, $Ym)
  {
   $MonthTable =  $VisMag = '';

   $Ym = substr(trim($Ym), 0,6);

// Adjust text case before search.
   $BodyName = ucfirst(strtolower(trim($BodyName)));

// Define names of Sun, Moon, planets and Pluto.
   $BodyNames = 'SunMoonMercuryVenusMarsJupiterSaturnUranusNeptunePluto';

// Define names of 15 asteroids.
   $AsteroidNames = 'CeresPallasJunoVestaHebeIrisFloraMetisHygieaEunomiaPsycheEuropaCybeleDavidaInteramnia';

// Error if body name is misspelled or not recognized. (Case INsensitive to user).
   if (strpos($AsteroidNames, $BodyName) === FALSE
   and strpos($BodyNames, $BodyName) === FALSE)
      {return FALSE;}

   if (strpos($AsteroidNames, $BodyName) !== FALSE)
  {
   $AsteroidFlag = TRUE;
   $ValidRangeStr = '_1800_to_2099';
  }
 else
  {
   $AsteroidFlag = FALSE;
   $ValidRangeStr = '-1600-to-2200';
  }

   $Y = substr($Ym, 0,4);
   $m = substr($Ym, 4,2);

// Determine if Gregorian common year or leap year and
// then compute number of days in given month/year.
   $leap = ((!($Y % 4) and ($Y % 100)) or !($Y % 400))? 1:0;

   $mDays = 30 - (($m < 8)? -1:1)*($m % 2) + floor($m/8)
          - (($m == 2)? $m:0) + (($m == 2 and $leap == 1)? 1:0);

// Compute JDTT for 1st of month at 00:00 TT  in  TZ=UT+00:00
   $JD00TT = Date_Time_Zone_to_JD_Zone("$Ym" . "01 00:00 +00:00", TRUE);

// Compute JD12 value for 1st day of month.
   $JD1st = floor($JD00TT + 0.5);

// Isolate leading JDTT value from string.
   $JD00TT = floatval($JD00TT);

// ERROR if JD00TT value is outside range from 1600 to 2200.
   if ($JD00TT < 2305447.5 or $JD00TT > 2524961.5) {return FALSE;}

// Read body data into working array and count total data file lines.
   $PlanetDataArray = file('data/geocen_equ_' . $BodyName . $ValidRangeStr . '.data');
   $PlanetDataCount = count($PlanetDataArray);

// Read each file line from the beginning and process ONLY data lines
// that begin with any digit (0 to 9).  All lines NOT beginning with
// a digit are treated as comment lines and ignored.
   for ($i=0;   $i < $PlanetDataCount;  $i++)
  {
   $DataLine = trim($PlanetDataArray[$i]);

   if (is_numeric(substr($DataLine, 0,1)))
  {
// Find data line where JD12TT value == JD1st.  This value
// becomes the first line of a monthly ephemeris table.
   $JD12TT = substr($DataLine, 0,7);

   if ($JD12TT == $JD1st)
  {
// Construct the 7-point interpolation
// tables for each of the statistics.
   for ($j = $i-3;   $j <= $i+$mDays+2;   $j++)
  {
// Get current data line.
   $DataLine = trim($PlanetDataArray[$j]);

// Get JD12TT value from beginning of current DataLine.
// (i0 to i6)
   $JD12TT = substr($DataLine, 0,7);

// Get RA in hours.
// (i7 to i15)
   $w = substr($DataLine, 7, 9);
   $RAHrs = substr($w, 0, 2) . '.' . substr($w, -7);

// Get declination in degrees.
// (i16 to i)
   $w = substr($DataLine, 16, 9);
   $DeclDeg = substr($w, 0, 3) . '.' . substr($w, -6);


// Use only 3 decimals for Moon distance.
   if ($BodyName == 'Moon')
  {
   $DistAUorKm = SPrintF('%1.3f', substr($DataLine, 25, 9)/1E3);
   $PhaseAngDeg = SPrintF('%1.5f',substr($DataLine, 34, strlen($DataLine))/1E5);

// Format phase angle left-padding with zeros.
   $PhaseAngDeg = substr("000$PhaseAngDeg", -9);
  }
else
  {
   $PhaseAngDeg = '';
   $DistAUorKm = substr($DataLine, 25, 2) . '.' . substr($DataLine, 27, 9);
  }

// Get visual magnitude, if applicable.
// (everything following the distance).
   $VisMag = '     ';
   if ($BodyName != 'Moon' and $BodyName != 'Sun')
  {
   $VisMag = SPrintF('%+1.1f', substr($DataLine, 36, strlen($DataLine))/10);
   $VisMag = substr("$VisMag       ", 0,7);
  }
   if($BodyName == 'Sun'){$VisMag = SPrintF('%+1.2f', 5*log10($DistAUorKm) - 26.74);}

   if ($BodyName == 'Moon')
      {$VisMag = '';}
   else
      {$VisMag = substr("$VisMag       ", 0,7);}

// Force day value to 2-digit format.
   $dd = SPrintF('%02d', $j-$i);
   if ($dd >= 0) {$dd = SPrintF('%02d', $dd+1);}
   if ($dd < 0 or $dd > $mDays) {$dd = 'xx';}

// Construct table line, normalize spacing and then append to table.
   $w = strip_wspace("$JD12TT $RAHrs $DeclDeg $DistAUorKm $PhaseAngDeg") . " $VisMag $dd";

   $w = str_replace(' 0', '  ', $w);

   $MonthTable .= "$w\n";

  } // End of for(j ...)
  } // End of  if(JD12TT ...)
  } // End of  if(is_numeric ...)
  } // End of for(i ...)

// Done.
   return trim($MonthTable);

} // End of Body_Month_Table(...)






// This function rounds up a standard time string
// to the nearest hour, minute or second.
// For example, 07:23:55 can be rounded up to 07:24
// or 09:40:17 could be rounded up to 10 hours.
//
// Standard time string includes colons "hh:mm:ss"

   function Round_HMS ($hhmmss, $HMSFilter='hms')
{
// Converting any colons into spaces.
   $hhmmss = str_replace(':', ' ', $hhmmss);

// Normalize spacing of h,m,s elements within the time string.
   $hhmmss = strip_wspace($hhmmss);

// Separate the given time elements.
   $HMSArray = preg_split("[ ]", $hhmmss);
   $ElementsCount = count($HMSArray);

// Fill in missing time elements with '00' before rounding up.
   $hh = ($ElementsCount >= 1)? SPrintF('%02d', $HMSArray[0]):'00';
   $mm = ($ElementsCount >= 2)? SPrintF('%02d', $HMSArray[1]):'00';
   $ss = ($ElementsCount >= 3)? SPrintF('%02d', round($HMSArray[2]),1):'00';

// Initialize HMS filter.
   $filter = strtoupper(trim($HMSFilter));

   if ($filter == 'HMS') {return "$hh:$mm:$ss";}

   if ($filter == 'HM')
   {
   if ($ss >= 30) {$mm++; $ss=0;}
   if ($mm > 59) {$hh++; $mm=0;}
   $hh = SPrintF('%02d', $hh);
   $mm = SPrintF('%02d', $mm);
   return "$hh:$mm";
   }

   if ($filter == 'H')
   {
   if ($ss >= 30) {$mm++; $ss=0;}
   if ($mm > 59) {$hh++; $mm=0;}
   $hh = SPrintF('%02d', $hh);
   $mm = SPrintF('%02d', $mm);
   if ($mm >= 30) {$hh++;}
   return "$hh";
   }

} // End of  Round_HMS(...)



/*
   This function computes the simple arithmetic mean
   of a given series of numbers entered in a string.
*/

   function Mean_Value($StringOfNumbers)
   {
    $w = preg_split('[ ]', strip_wspace($StringOfNumbers));
    $wCount = count($w);
    $mean = array_sum($w) / $wCount;
    return $mean;
   }



/*
   This function computes the rectangular coordinates corresponding
   to a given set of spherical coordinates.

   The RA can be either decimal degrees or decimal hours.  For hours,
   simply attach 'h' to the end of the RAHDeg string.  Otherwise, the
   default degrees will be assumed.

*/

   function RA_Decl_Dist_to_XYZ ($RAHorDeg, $DeclDeg, $distance)
{

// Get value of RA (w) and if hours, then convert to degrees.
// Otherwise, assume (RA) is already in degrees (default).
   $w = trim($RAHorDeg);  if (strtolower(substr($w, -1)) == 'h') {$w *= 15;}

// Convert working RA and declination
// degree values into radians.
   $RA   = deg2rad($w);
   $decl = deg2rad($DeclDeg);

// Distance can be in any units:
// km, miles, AU, LY, etc.
   $R = $distance;

// Compute rectangular XYZ-coordinates
// in the same units as distance (R).
   $X = $R * cos($RA) * cos($decl);
   $Y = $R * sin($RA) * cos($decl);
   $Z = $R * sin($decl);

   return "$X $Y $Z";
}



   function Lng_Lat_R_to_XYZ ($LngDeg, $LatDeg, $R)
{
   if (is_numeric($LngDeg) and is_numeric($LatDeg) and is_numeric($R))
  {
// Convert longitude and latitude
// from degrees into radians.
   $lng = deg2rad($LngDeg);
   $lat = deg2rad($LatDeg);

// Compute rectangular XYZ-coordinates
// in the same units as distance (R).
   $X = $R * cos($lng) * cos($lat);
   $Y = $R * sin($lng) * cos($lat);
   $Z = $R * sin($lat);

   return "$X $Y $Z";
   }
   return FALSE;
}





   function Is_Valid_Body ($BodyName)
   {
// Adjust text case before search.
   $BodyName = ucfirst(strtolower(trim($BodyName)));

// Define names of Sun, Moon, planets, Pluto and 15 asteroids.
   $BodyNames = ' Sun Moon Mercury Venus Mars Jupiter Saturn Uranus Neptune Pluto Ceres Pallas Juno Vesta Hebe Iris Flora Metis Hygiea Eunomia Psyche Europa Cybele Davida Interamnia ';

// Error if body name is misspelled or not recognized. (NOT case sensitive to user).
   if (strpos($BodyNames, ' ' . $BodyName . ' ') === FALSE) {return FALSE;}

   return TRUE;
   }



/*
============================================================
Temperature scale interconversion function.

PHP VERSION : 5.4.7
AUTHOR      : Jay Tanner - 2014


This function converts from any given temperature
scale into the equivalent temperature on any one
of four other recognized (F,C,K,R) scales.

The scale symbols are:
F = Fahrenheit
C = Celsius
K = Kelvin  = Degrees Celsius above absolute zero.
R = Rankine = Degrees Fahrenheit above absolute zero.

The scale symbols are NOT case sensitive.

The conversion is performed by first converting the
input temperature value into its equivalent on the K
scale, then the K scale value is converted into the
given output scale.

-------
EXAMPLE
To convert from 123.45 C into its equivalent
on the R scale:

print Temp_To_Temp("-123.45 C", "R"); // Result = 269.46


------
ERRORS
An error returns FALSE if a scale symbol is not recognized.
============================================================
*/


   function Temp_To_Temp ($FromTempStr, $ToScaleStr)
{
// -------------------------------------
// Read input argument value and scales.

   $w = trim($FromTempStr);   if ($w == '') {$w = "0";}
   $T = floatval($w);

   $FromScale = substr($w, -1);
   $ToScale = trim($ToScaleStr);

// ---------------------------------------------------
// Convert input ($FromScale) into K scale equivalent.

   switch (strtoupper(trim($FromScale)))
  {
   case "F" : $K = (5 * ($T - 32) / 9) + 273.15; break;
   case "C" : $K = $T + 273.15; break;
   case "K" : $K = $T; break;
   case "R" : $K = (5 * ($T - 491.67) / 9) + 273.15; break;
   default  : $K = FALSE; break;
  }

// --------------------------
// ERROR if invalid FromScale

   if ($K === FALSE) {return $K;}

// --------------------------------------------------
// Convert K value into output ($ToScale) equivalent.

   switch (strtoupper(trim($ToScale)))
  {
   case "F" : $T = (9 * ($K - 273.15) / 5) + 32; break;
   case "C" : $T = $K - 273.15; break;
   case "K" : $T = $K; break;
   case "R" : $T = (9 * ($K - 273.15) / 5) + 491.67; break;
   default  : $T = FALSE; break;
  }

   return $T;

} // End of Temp_To_Temp(...)




/*
 ===========================================================================
 This function interconverts between any of sixteen length or
 distance units.  The unit symbols are NOT case sensitive.

 ----------
 ARGUMENTS:
 $LengthAndUnits = Numerical value and units symbol of length to convert.
                   NOTE: There MUST be at least one space between the
                   number value and the units symbol in the string.
                   The unit symbols are NOT case sensitive.

 $ToUnits = The units symbol corresponding to the computed output value.

 -------
 ERRORS:
 On error FALSE is returned.
 An error occurs if the length value is non-numeric or either
 of the input or output length units symbols are unrecognized.

 ------------------------------------------------------------
 THE 16 LENGTH UNITS SYMBOLS RECOGNIZED BY THIS FUNCTION ARE:

 SYMBOL      UNITS

   A         Angstroms
   nm        Nanometers
 um or u     Micrometers (or Microns)
   mm        Millimeters
   cm        Centimeters
   in        Inches
   ft        Feet
   yd        Yards
   m         Meters
   km        Kilometers
   mi        Miles (statute)
   nmi       Nautical miles
   LS        Light Seconds
   AU        Astronomical Units
   LY        Light Years (Based on Julian year of 365.25 days)
   pc        Parsecs

   ===========================================================================
*/

   Function Length_To_Length ($LengthAndUnits, $ToUnits)
{
// Define conversion constants for internal use only.
// These constants could also be defined externally
// for universal use by any programs that need them.
// This function was designed to be transportable.
   $M_PER_A   = 0.0000000001;
   $M_PER_NM  = 0.000000001;
   $M_PER_UM  = 0.000001;
   $M_PER_MM  = 0.001;
   $M_PER_CM  = 0.01;
   $M_PER_IN  = 0.0254;
   $M_PER_FT  = 0.3048;
   $M_PER_YD  = 0.9144;
   $M_PER_M   = 1;
   $M_PER_KM  = 1000;
   $M_PER_MI  = 1609.344;
   $M_PER_NMI = 1852;
   $CLIGHT    = 299792458;
   $M_PER_AU  = 149597870691;
   $M_PER_LY  = 9460730472580800;
   $M_PER_PC  = 30856775876793114.6633;

// Parse input value and separate the numerical value from the symbol.
// There MUST be at least one space between the number and the symbol.
// FALSE is returned on error if no space character detected at all or
// the given numeric value is not a valid number.
   $w = trim($LengthAndUnits); if (strpos($w, " ") === FALSE){return FALSE;}

   list($x, $FromUnits) =
        preg_split("[ ]", preg_replace("/\s+/", ' ', $w));

   if (!is_numeric($x)) {return FALSE;}

// Get input units symbol.  Unit symbols are NOT case sensitive.
   $FromUnitsSymbol = strtoupper(trim($FromUnits));

// Determine which conversion factor to use
// according to the units symbol argument.

   switch ($FromUnitsSymbol)
  {
   case "A"  : $f = $M_PER_A;   break;
   case "NM" : $f = $M_PER_NM;  break;
   case "UM" : $f = $M_PER_UM;  break;
   case "U"  : $f = $M_PER_UM;  break;
   case "MM" : $f = $M_PER_MM;  break;
   case "CM" : $f = $M_PER_CM;  break;
   case "IN" : $f = $M_PER_IN;  break;
   case "FT" : $f = $M_PER_FT;  break;
   case "YD" : $f = $M_PER_YD;  break;
   case "M"  : $f = $M_PER_M;   break;
   case "KM" : $f = $M_PER_KM;  break;
   case "MI" : $f = $M_PER_MI;  break;
   case "NMI": $f = $M_PER_NMI; break;
   case "LS" : $f = $CLIGHT;    break;
   case "AU" : $f = $M_PER_AU;  break;
   case "LY" : $f = $M_PER_LY;  break;
   case "PC" : $f = $M_PER_PC;  break;
   default   : $f = FALSE;      break;
  }

// Error if conversion factor is FALSE, which
// means that the units symbol was not found
// in the table of valid symbols.
   if ($f === FALSE) {return $f;}

// Otherwise, compute the equivalent meters value
// by multiplying the value of the argument by
// the conversion factor $f.
   $m = $x * $f;

// *********************************************************
// AT THIS POINT INPUT VALUE HAS BEEN CONVERTED INTO METERS.
// NEXT STEP IS TO CONVERT THE METERS INTO THE OUTPUT UNITS.
// *********************************************************

// Read the output units symbol argument.  The unit symbols
// are NOT case sensitive.
   $ToUnitsSymbol = strtoupper(trim($ToUnits));

// Determine the conversion factor to use from the units
// symbol argument.

   switch ($ToUnitsSymbol)
  {
   case "A"  : $f = $M_PER_A;   break;
   case "NM" : $f = $M_PER_NM;  break;
   case "UM" : $f = $M_PER_UM;  break;
   case "U"  : $f = $M_PER_UM;  break;
   case "MM" : $f = $M_PER_MM;  break;
   case "CM" : $f = $M_PER_CM;  break;
   case "IN" : $f = $M_PER_IN;  break;
   case "FT" : $f = $M_PER_FT;  break;
   case "YD" : $f = $M_PER_YD;  break;
   case "M"  : $f = $M_PER_M;   break;
   case "KM" : $f = $M_PER_KM;  break;
   case "MI" : $f = $M_PER_MI;  break;
   case "NMI": $f = $M_PER_NMI; break;
   case "LS" : $f = $CLIGHT;    break;
   case "AU" : $f = $M_PER_AU;  break;
   case "LY" : $f = $M_PER_LY;  break;
   case "PC" : $f = $M_PER_PC;  break;
   default   : $f = FALSE;      break;
  }

// Error if conversion factor is FALSE, which
// means that the units symbol was not found
// in the valid symbols table.
   if ($f == FALSE) {return FALSE;}

// Otherwise, return the equivalent of the given
// meters value by dividing the value of the
// argument by the conversion factor $f.
   return $m / $f;

} // End of  Length_To_Length(...)


/*
   ------------------------------------------------------------
   This function computes the theoretical blackbody temperature
   at any given distance from the sun in AU.

   $DistAU      = Distance in AU
   $TScaleKFC   = 'K|F|C|R' - Default = K
   $SolarConstW = 1360.8 W/m²/s = Default

*/

   function Temp_KFCR_at_Dist_AU ($DistAU, $TScaleKFCR="K", $SolarConstW=1360.8)
{
   $KFCR = strtoupper(substr(trim($TScaleKFCR), -1));

   if ($KFCR == "") {$KFCR = "K";}

// Error if scale symbol is not recognized.  Must be K|F|C|R
   if ($KFCR != "K" and $KFCR != "F" and $KFCR != "C" and $KFCR != "R")
  {
   return FALSE;
  }

// Compute solar radiant energy flux at DistAU.
   $e = $SolarConstW / $DistAU / $DistAU;

// Compute corresponding temerature on K,F,C,R scales.
   $K = sqrt(sqrt($e / 0.000000056704));
   $F = temp_to_temp("$K K", "F");
   $C = temp_to_temp("$K K", "C");
   $R = temp_to_temp("$K K", "R");

   $e = SPrintF("%4.6f", $e);
   $D = SPrintF("%3.3f", $DistAU);

// Return temperature on given scale.
   switch ($KFCR)
  {
   case "K" : $T = $K; break;
   case "F" : $T = $F; break;
   case "C" : $T = $C; break;
   case "R" : $T = $R; break;
  }
   return SPrintF("%4.3f", $T) . " $KFCR";
}





/*
   This function computes the angular diameter of a
   sphere viewed at any distance from its center.

   The equation used accounts for the 3D perspective
   of the curved surface of the sphere vs distance.

   Radius and distance must be measured in same units.

   Returned angle is in degrees.
*/

   function Ang_Diam_at_Dist ($radius, $distance)
  {
   $R = floatval($radius);
   $D = floatval($distance);

// Return FALSE on error, if R*D equates to zero.
   if ($R*$D == 0) {return FALSE;}

// Return angular diameter of sphere, in degrees.
   return rad2deg(2*acos(SqRt($D*$D - $R*$R) / $D));
  }


/*
   ===================================================================
   This function computes the full astronomical JD (Julian Day) number
   on the TT scale for any given local date, time and time zone over
   the 600 year span from 1600 to 2200.

   Times are assumed to be standard zone times.  No daylight saving or
   summer time adjustment is applied.

   ARGUMENTS:
   $YmdHMSTZString = Local Date/Time/Zone string (Ex. "19490520 18:40 -05:00")
   $dTFlag = Delta T flag = (Default = FALSE).
   if TRUE, then apply Delta T to input time.

   Example Date/Time/Zone argument string for
   1949 May 20 at 20:52 (8:52 PM) in TZ -05:00
   = "19490520 20:52 -05:00"
   For UT, TZ = +00:00 = Default

   If dTFlag === FALSE (default), given time is assumed to be TT.
   If dTFlag TRUE, given time is converted to TT by applying Delta T.
   NOT TRUE = FALSE = Anything other than TRUE.
   ===================================================================
*/


   function Date_Time_Zone_to_JD_Zone ($DateTimeString, $DeltaTFlag=FALSE)
{
// Transparently normalize spacing between given date/time elements.
   $w = preg_split("[ ]", strip_wspace($DateTimeString));

/* Count date/time arguments (1 to 3).  Must
   be at least the date value, followed by
   optional time and UT time zone offset.
   Local Standard Time - TZ = UT

   Example Date/Time Strings
     Ymd    hhmm  TZhhmm
   20170520 08:52 -05:00   = 2017 May 20 at 08:52 in TZ -05 (New York, USA)
   20170520 08:52          = 2017 May 20 at 08:52 in TZ -00 (Greenwich, UK)
   20170520                = 2017 May 20 at 00:00 in TZ -00 (Greenwich, UK)
*/
   $ArgCount = count($w);

// Determine Delta T boolean flag setting.
// FALSE = Time is UT,  TRUE = Time is TT
   $dTFlag = ($DeltaTFlag === TRUE)? TRUE:FALSE;

// Parse date/time elements.
   $Ymd = intval($DateTimeString);
     $Y = substr($Ymd, 0,4);
     $m = substr($Ymd, 4,2);
     $d = substr($Ymd, 6,2);

// Parse time sub-string.
   $hhmmss   = ($ArgCount >= 2)? $w[1]:"00:00:00";
   $TimeDays = HMS_to_Hours($hhmmss)/24;

// Parse time zone sub-string.
   $TZhhmm = ($ArgCount >= 3)? $w[2]:"+00:00";
   $TZDays = HMS_to_Hours($TZhhmm,TRUE)/24;
    $TZHMS = Hours_to_HMS($TZhhmm, 0, TRUE, ":");
   $TZhhmm = substr($TZHMS, 0, strlen($TZHMS)-3);

// Apply Delta T, if indicated (TRUE).
   $DeltaTDays = 0;
   if ($DeltaTFlag === TRUE)
  {
   $DeltaTDays = Delta_T($Ymd)/86400;
  }

// Compute JD number for 00:00:00 TT of date.
   $JD00TT = GregorianToJD($m, $d, $Y) - 0.5;

// Compute full astronomical JD number for all date/time elements.
   $JDTT = SPrintF("%7.8f", $JD00TT + $TimeDays - $TZDays + $DeltaTDays);

// Return the JD value and the standard UT time zone offset.
   return "$JDTT $TZhhmm";

} // End of Date_Time_Zone_to_JD_Zone(...)



/*
   This function generates the HTML IMG code to display
   the lunar phase image corresponding to the given
   phase angle to the nearest degree and optional
   parallactic angle.
*/

   function Lunar_Phase_Image($PhaseAngDeg, $ParallacticAngDeg=0)
{
   $PhaseImageName = SPrintF("%03d", round(floatval($PhaseAngDeg), 0));

   $ParallAngDeg = floatval($ParallacticAngDeg) . "deg";

   return "<img src=\"lunar_phases/$PhaseImageName.webp\" title=\"Phase $PhaseImageName deg\" style=\"transform:rotate($ParallAngDeg);\">";
}





   function Azim_Alt2 ($JDTT, $RAHMS, $DeclDMS, $LngDMS, $LatDMS)
{

// Read RA/Decl as decimal degrees
   $RAHr = HMS_to_Hours($RAHMS);
   $DeclDeg = DMS_to_Deg($DeclDMS);

// Read lat/lng as decimal degrees
   $LatDeg = DMS_to_Deg($LatDMS);
   $LngDeg = DMS_to_Deg($LngDMS);

// Compute local true sidereal time.
   $LSTTrueDeg = Mean_Sid_Time($JDTT, $LngDeg) + dEquinox_2000B($JDTT);

// Compute local hour angle of body.
   $H = $LSTTrueDeg - $RAHr*15;

// Compute azimuth measured from eastward from north 0 to 360 degrees (North = 0 degrees).
   $AzimDeg = 180 + atan2_d(sin_d($H), (cos_d($H)*sin_d($LatDeg) - tan_d($DeclDeg)*cos_d($LatDeg)));
   $AzimDeg -= (($AzimDeg >= 360)? 360:0);

// Compute altitude relative to local horizon.  Negative = Below horizon.
   $SinAlt = sin_d($LatDeg)*sin_d($DeclDeg) + cos_d($LatDeg)*cos_d($DeclDeg)*cos_d($H);
   $AltDeg = asin_d($SinAlt);

   return "$AzimDeg $AltDeg";
}



   function Random_Date_Time()
{
   $Y  = mt_rand(1600, 2200);
   $m  = SPrintF("%02d", mt_rand(1, 12));
   $d  = SPrintF("%02d", mt_rand(1, 28));
   $hh = SPrintF("%02d", mt_rand(0, 23));
   $mm = SPrintF("%02d", mt_rand(0, 59));
   $ss = SPrintF("%02d", mt_rand(0, 59));

   return "$Y$m$d $hh:$mm:$ss";
}







   function Lunar_Phase_vs_JD12TT_Table($Ym)
{
   $Y = substr($Ym, 0,4);
   $m = intval(substr($Ym, 4,2));

// Determine days in given month.
   $mDays = Cal_Days_In_Month(CAL_GREGORIAN, $m, $Y);

// Compute ISO integer-encoded date value.
   $DateValue = $Y*10000 + $m*100 + 1;

// Compute JD12TT for 1st day of month.
   $JD12TT1st = intval(Date_Time_Zone_to_JD_Zone("$DateValue 12"));

// Define JD12TT table endpoints.
   $JD12TTStart = $JD12TT1st - 3;
   $JD12TTEnd   = $JD12TT1st + $mDays + 2;

// Read lunar data file into working array.
   $w = file("data/geocen-equ-Moon-1600-to-2200.data");

// Count total table text lines.
   $wCount = count($w);

// Find table line starting with JD12TTStart value.
   for($i=0;   $i < $wCount;   $i++)
  {
   $JD12TT = substr(trim($w[$i]), 0,7);

   if ($JD12TT == $JD12TTStart) {break;}
  }

// Resume here after break.
// i points to start of table.


// Construct lunar phase vs JD12TT
// interpolation table for month.
   $PhaseAngVsJD12TTTable = "";

   for ($j=0;   $j < $mDays+6;   $j++)
  {
   $u = trim($w[$i+$j]);
   $JDTT = substr($u, 0, 7);
   $PhaseAngDeg = SPrintF("%3.5f", substr($u, -8)/100000);
   $PhaseAngDeg = substr("         $PhaseAngDeg", -9);

   $PhaseAngVsJD12TTTable .= "$PhaseAngDeg $JDTT\n";
  }

// Done.  Return phase interpolation table.
   return trim($PhaseAngVsJD12TTTable);

} // End of Lunar_Phase_vs_JD12TT_Table(...)






/*
   Search for given phase angle and return
   JDTT corresponding to date/time TT.

   Then use inverse JD function to compute
   local date/time of given phase angle.

   CAN'T DO NEW MOON YET

*/

   function Moon_Phase_Date_Time($PhaseAngDeg, $Ym, $TZhhmm="+00:00", $DeltaTFlag=FALSE)
{
// Construct phase interpolation table.
   $wTableStr = Lunar_Phase_vs_JD12TT_Table($Ym);

   $dTFlag = ($DeltaTFlag !== FALSE)? TRUE:FALSE;

// Split table into working array.
   $wArray = preg_split("[\n]", $wTableStr);
   $wArrayCount = count($wArray);

// Find center point to within 15 degrees.
// Search from i=3 to mDays-1
// Create work table
   $wTableStr = "";

   for($i=3;   $i < $wArrayCount-3;   $i++)
   {
   $w = floatval(trim($wArray[$i]));
   $diff = abs($w - $PhaseAngDeg);
   if ($diff <= 15) {break;}
   }

   for($j = $i-3;   $j <= $i+3;   $j++)
   {
   $wTableStr .= $wArray[$j] . "\n";
   }

// Interpolate JDTT for phase angle value.
   $JDTT = lagrange_interp($wTableStr, $PhaseAngDeg);

// Compute Date/Time TT of given phase.
   $DateTimeTT = jd_zone_to_date_time_zone("$JDTT $TZhhmm", $dTFlag);

   return $DateTimeTT;
}






// #########################################
// #########################################
// #########################################
// #########################################
// #########################################









/* ------------------------------------------------------
   Level 1

   DEPENDENCY:
   JD_to_Date_Time(...)
   Lunar_Phase_Table(...)
   Swap_LaGrange_Data(...)
   LaGrange_Interpolate(...)
   Deg_Mod_360(...)

   Given the astronomical JD number for any given moment,
   compute the corresponding geocentric lunar phase angle
   from 0 to +360 degrees.


*/

   function Lunar_Phase_Angle($JDTT)
{
// Get year/month substring for which to compute table.
   $Ym = substr(JD_to_Date_Time($JDTT), 0,6);

// Get 1st table: "PhaseAngDeg JD12TT"
   $PhaseTable1 = Lunar_Phase_Table($Ym);

// Create inverted 2nd table: "JD12TT PhaseAngDeg+
   $PhaseTable2 = Swap_LaGrange_Data($PhaseTable1);

// Given JD, interpolate the corresponding PhaseAngDeg
// within PhaseTable2
   $PhaseAngDeg = LaGrange_Interpolate($PhaseTable2, $JDTT);

// Modulate to fall in the range from 0 to 360 degrees.
   $PhaseAngDeg = sprintf("%1.5f", Deg_Mod_360($PhaseAngDeg));

// Done.
   return $PhaseAngDeg;
}






/* ------------------------------------------------------
   Level 1

   DEPENDENCY:
   File: (Lunar_Phases-1600-to-2200.data)
   Date_Time_to_JD(...)

*/
   function Lunar_Phase_Table($YmStr)
{
   global $DataPath;

   $LunarPhaseTable = "";

// Parse and extract the date elements.
   $Ym = trim($YmStr);
   $Y  = substr($Ym, 0,4);
   $m  = intval(substr($Ym, 4,2));

// Load lunar phases data file and count the file lines.
   $PhaseDataArray = file("../data/Lunar-Phases-1600-to-2200.data");
   $PhaseDataArrayCount = count($PhaseDataArray);

// ---------------------------------
// Compute JD12 for start/end dates.
   $JD12Start = Date_Time_to_JD($Ym . "01 12:00");

// Check for bad date and return FALSE if outside valid tabular range.
   if ($JD12Start === FALSE or $JD12Start < 2305448 or $JD12Start > 2524958)
       {return FALSE;}




// --------------------------------------------------------------
// Compute number of days in month, accounting for any leap year.
   $mDays = substr("312831303130313130313031", 2*($m-1), 2);

   if ($m == 2)
      {
       $mDays += (((!($Y % 4) and ($Y % 100)) or !($Y % 400))? 1:0);
      }


// ------------------------------
// Compute start/end JD12 values.
   $JD12Start -= 2;
   $JD12End = $JD12Start + $mDays + 3;

// --------------------------------------------------
// Search data array for JD12 start value and extract
// month sub-table equal to length of given month
// plus an extra day at each end of the month.

   for ($i=0;   $i < $PhaseDataArrayCount;   $i++)
  {
   $PhaseDataLine = trim($PhaseDataArray[$i]);

   if (intval($PhaseDataLine) > 0)
      {
//       list($PhaseAngDeg, $JD12i) = preg_split("[ ]", $PhaseDataLine);
$JD12i = substr($PhaseDataLine, -7);
$PhaseAngDeg = substr($PhaseDataLine, 0, strlen($PhaseDataLine)-7);


       if ($JD12i >= $JD12Start-1 and $JD12i <= $JD12End+1)
          {
           $LunarPhaseTable .= (($PhaseAngDeg/100000) . " $JD12i\n");
          }
      }
  }

// --------------------------------------------------------
// Get month table array into working array.  In this case,
// the data delimiters are the new-line (\n) breaks at the
// end of each text data line in the table.  Then adjust
// the phases to fall in the range from 0 to 810 degrees
// for subsequent interpolative continuity over the span.

   $wArray = preg_split("[\n]", trim($LunarPhaseTable));

   $wArrayCount = count($wArray);

   $LunarPhaseTable = "";

   $flag = $PrevPhase = FALSE;

   for ($j=1;   $j < $wArrayCount-1;   $j++)
  {
   list ($PhaseAngDeg, $JD12) = preg_split("[ ]", $wArray[$j]);

   if (floatval($wArray[$j]) - floatval($wArray[$j-1]) < 0 or $flag === TRUE)
  {
   $PhaseAngDeg += 360;
   $PhaseAngDeg += ($PrevPhase - $PhaseAngDeg > 0 and $flag === TRUE)? 360:0;
   $PrevPhase = $PhaseAngDeg;
   $flag = TRUE;
  }
   $PhaseAngDeg = sprintf("%1.5f", $PhaseAngDeg);
   $LunarPhaseTable .= "$PhaseAngDeg $JD12\n";
  }
   return trim($LunarPhaseTable);
}











   function Date_Time_to_JD ($Ymd_HMS)
{
// Strip all white space from date/time string.
   $w = preg_replace("/\s+/", " ", trim($Ymd_HMS));

// If no time string, then attach default zero time.
   if (strpos($w, " ") === FALSE) {$w .= " 00:00:00";}

// Split date/time string into separate variables.
   list($Ymd, $HMS) = preg_split("[ ]", $w);

// Convert HMS time elements string into HMS array and
// then count the colon-delimited time elements.
   $HMS = preg_split("[\:]", $HMS);
   $HMSCount = count($HMS);

// Fetch the individual time element (hh,mm,ss) values.
   $hh = ($HMSCount >= 1)? $HMS[0]:0;
   $mm = ($HMSCount >= 2)? $HMS[1]:0;
   $ss = ($HMSCount >= 3)? $HMS[2]:0;

// Parse the ISO integer-encoded date argument and extract
// the individual date arguments (Y,m,d) from within it.
   $Y = floor($Ymd / 10000);
   $m = floor(($Ymd - 10000*$Y) / 100);
   $d = $Ymd - 10000*$Y - 100*$m;

// Check date for validity (must be in the range from 1600 to 2400).
   if(checkdate($m,$d,$Y) === FALSE or $Y < 1600 or $Y > 2400) {return FALSE;}

// Compute Gregorian JD number for 12h TT of
// date and then the JD00 value from it.
   $JD00 = GregorianToJD($m, $d, $Y) - 0.5;

// Compute the JD fraction corresponding
// to the given time (hh,mm,ss) elements.
   $sh = bcmul("3600", $hh, 20);
   $sm = bcmul("60",   $mm, 20);
   $JDFrac = bcdiv(bcadd($sh, bcadd($sm,$ss, 20), 20),"86400", 20);

// Apply fraction of day to JD value and
// round off to up to 16 decimals max.
   $JD = bcadd(bcadd($JD00, $JDFrac, 20), "0.00000000000000005", 16);

// Trim off any redundant zeros and/or decimal point. Done.
   return rtrim(rtrim($JD, "0"), ".");

} // End of  Date_Time_to_JD(...)



/*
   =====================================================================
   Level 0:

   This function serves as an inverse Gregorian JD function.  Given the
   general JD number, this function returns the date and time resolved
   to the given number of decimals from 0 to 3.
   =====================================================================
*/

   function JD_to_Date_Time ($JDArg, $ssDecimals=0)
{
// Read JD number argument.
   $JD = trim($JDArg);

// Error if argument is non-numeric.
   if (!is_numeric($JD)) {return FALSE;}

// If JD is an integer value, then attach ".0" to end of it.
   if (strpos($JD, ".") === FALSE) {$JD .= ".0";}

// Compute JD12 value for date corresponding to JD argument.
   $JD12 = floor($JD + 0.5);

// Compute Gregorian date corresponding to the JD argument.
   list($m, $d, $Y) = preg_split("[\/]", JDtoGregorian($JD12));

// Force month and day values into 2-digit format.
   $m = sprintf("%02d", $m);
   $d = sprintf("%02d", $d);

// Construct integer encoded date from date elements.
   $Ymd = "$Y$m$d";

// Isolate fractional part of JD that will be converted into time.
   $j = bcsub($JD, "0.5", 16);
   $hours = bcmul("24", "0" . substr($j, strpos($j, "."), strlen($j)), 16);

// Compute hours value.
   $hh = floor($hours);

// Compute minutes value.
   $minutes = 60*($hours - $hh);
   $mm = floor($minutes);

// Compute seconds value
   $seconds = 60*($minutes - $mm);

// Format (hh, mm, ss) values for output.
   $hh = sprintf("%02d",  $hh);
   $mm = sprintf("%02d",  $mm);
   $ss = sprintf("%1.$ssDecimals" . "f",$seconds); if($ss < 10) {$ss = "0$ss";}

// Done.
   return "$Ymd $hh:$mm:$ss";

} // End of  JD_to_Date_Time(...)




/*
   ========================================================================
   Level 0

   This function performs LaGrange interpolation within an XY data table
   given as a string of matching space-delimited data pairs.

   ========================================================================
*/

   function LaGrange_Interpolate ($XYDataTable, $xArg)
{

// -----------
// Initialize.

   $XDataStr = $YDataStr = "";

// --------------------------------------------------------------
// Read and split XY data pairs into a work array.  In the array,
// even number indices = X-data, odd number indices = Y-data.

   $XY = preg_split("[ ]", preg_replace("/\s+/", " ", trim($XYDataTable)));

// ---------------------------------------------
// Count the total number of data elements. This
// value should always be an even number.

   $TotalDataCount = count($XY);

// ------------------------------------------------------------
// Number of data pairs.  This value should be an integer value
// exactly equal to 1/2 the total number of data points. If not,
// then there is an odd mismatched data point.

   $n = $TotalDataCount / 2;

// -----------------------------------------------------------
// Return error message if data vector element count mismatch.
// For every X value there must be a corresponding Y value or
// an XY data count mismatch error occurs.  An error will also
// occur if insufficient data.  There must be at least two XY
// data points given.

   if ($TotalDataCount < 4 )
      {return "ERROR: There must be at least two XY data pairs.";}

   if ($n != floor($n + 0.5))
      {return "ERROR: XY Data Count Mismatch. Odd data element.";}

// ------------------------------------------------------------
// Compute number of XY data pairs.  This value is exactly half
// the value of the total number of data elements.

   $n = $TotalDataCount / 2;

// -------------------------------------------------------
// Construct separate XY data strings from the array data.
// The XY data strings should each contain the same number
// of data elements.

   for($i=0;   $i < $TotalDataCount;   $i+=2)
      {
       $XDataStr .= $XY[$i]   . " ";
       $YDataStr .= $XY[$i+1] . " ";
      }

// --------------------------------------------------------------
// Split the created XY data vector strings into matching indexed
// arrays.  For every X value there must be a matching Y value
// and no two X values can be identical.

   $X = preg_split("[ ]", trim($XDataStr));
   $Y = preg_split("[ ]", trim($YDataStr));

// ----------------------------------------
// Read X argument for which to interpolate
// the Y value from the given XY data.

   $x = trim($xArg);  if ($x == "") {$x = "0";}

// -----------------------------------
// Initialize Y summation accumulator.
   $y = 0.0;

// -----------------------------------------------------
// Compute Lagrangian product (Li) for given X argument.

   for ($i=0;   $i < $n;   $i++)
       {
        $Li = 1.0;

        for ($j=0;   $j < $n;   $j++)
            {
             if ($j != $i) // Skip this cycle when j == i
                {
                 $Li = ($Li * ($x - $X[$j])) / ($X[$i] - $X[$j]);
                }
            } // Next j

//      --------------------------------------
//      Accumulate sum of Yi polynomial terms.

        $y += ($Y[$i] * $Li);

       } // Next i

   return $y;

} // End of  LaGrange_Interpolate(...)




/*
   ========================================================================
   Level 0

   This function simply swaps all matching XY data pairs held in a
   space delimited string argument. It was designed as a companion
   to the Lagrange_Interpolate() function to easily facilitate the
   swapping of the XY data interpolation roles.

   The data trend can be linear or non-linear, however
   only 2 data pairs are needed for any linear trend.

   ERRORS:
   On error FALSE is returned.  An error occurs if
   odd number of data elements or < 4 elements.
   ========================================================================
*/

   function Swap_LaGrange_Data ($XYDataStr)
{

// -------------------------------------------------------------------------
// Read space-delimited data string and split items into a sequential array.
   $XYArray = preg_split("[ ]", preg_replace("/\s+/", " ", trim($XYDataStr)));

   $XYArrayCount = count($XYArray);

   if ($XYArrayCount % 2 != 0 or $XYArrayCount < 4) {return FALSE;}

   $XYPairsCount = $XYArrayCount / 2;

// -------------------------------------------------------
// Construct and return output string of swapped XY pairs.
   $YXDataPairsStr = "";

   for ($i=1;   $i < $XYArrayCount;   $i+=2)
       {
        $YXDataPairsStr .= $XYArray[$i] . " " . $XYArray[$i-1] . "\n";
       }

   return trim($YXDataPairsStr);
}


/*
   ========================================================================
   Level 0

   This function modulates a negative or positive angle to fall within the
   range from 0 to +360 degrees to help handle angles that fall outside
   the standard circular values.  For example, -5 degrees would be
   converted to +355 degrees or 417 degrees would be converted
   to 57 degrees, etc.

   This function uses arbitrary precision arithmetic.

   ARGUMENTS:
   $DegStr = Degrees argument as a numerical string.

*/

   function Deg_Mod_360 ($DegStr)
  {
   $SignVal = ($DegStr < 0)? -1:1;

   $a = bcmul($SignVal, "$DegStr", 20);
   $b = bcmul($SignVal, bcsub($a, bcmul(360, bcdiv($a, 360)), 20),20);

   return rtrim(rtrim(bcadd($b, (($b < 0)? 360:0), 16), "0"), ".");

  } // End of  Deg_Mod_360()





/*
  ========================================================================
  Level 0

  This function computes the theoretical astronomical delta-T estimate
  in seconds for any dates between -2000 and 3000, based on the NASA
  polymomial expressions for Delta T

  NASA Ref:
  http://eclipse.gsfc.nasa.gov/LEcat5/deltatpoly.html


  ARGUMENTS:
  Y = Year number (-2000 BC to 3000 AD) (Neg=BC)(no year zero)
  mNum = Month number(1 to 12)

  ERRORS:
  FALSE is returned on error.  An error results if either
  the year or month number is invalid. A zero year value
  will return an error.
  ========================================================================
*/

   function Delta_T_Estimate ($Y, $mNum=1)
{

// --------------------------------------------------------
// The delta T value returns FALSE for invalid argument(s).

   $dT = FALSE;

// ----------------------------------------------
// ERROR if year value is non-numeric. It must be
// a number in the range from -2000 to +3000.

   if (!is_numeric($Y)) {return FALSE;}

// ------------------------------------------
// ERROR if year value is invalid. It must be
// in the range from -2000 to +3000.

   if ($Y < -2000 or $Y == 0 or $Y > 3000) {return FALSE;}

// -------------------------------------------------------
// ERROR if month value is outside the range from 1 to 12.
   if ($mNum < 1 or $mNum > 12) {return FALSE;}

// -----------------------------------------------------------
// If a negative (BC) year is given , it is converted into the
// corresponding mathematical year by adding 1.
//
// The year 100 BC (-100), becomes the mathematical year -99.

   $y = $Y + (($Y < 0)? 1:0) + ($mNum - 0.5)/12;


   if ($Y >= -2000 and $Y < -500)
      {
       $u  = ($Y - 1820) / 100;

       return 32*$u*$u - 20;
      }


   if ($Y >= -500 and $Y < 500)
      {
       $u  = $y/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 10583.6 - 1014.41*$u
                      +   33.78311*$u2
                      -    5.952053*$u3
                      -    0.1798452*$u4
                      +    0.022174192*$u5
                      +    0.0090316521*$u6;
      }


   if ($Y >= 500 and $Y < 1600)
      {
       $u  = ($y - 1000)/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 1574.2 - 556.01*$u
                     +  71.23472*$u2
                     +   0.319781*$u3
                     -   0.8503463*$u4
                     -   0.005050998*$u5
                     +   0.0083572073*$u6;
      }


   if ($Y >= 1600 and $Y < 1700)
      {
       $u  = $y - 1600;

       return 120 - 0.9808*$u - 0.01532*$u*$u + $u*$u*$u/7129;
      }


   if ($Y >= 1700 and $Y < 1800)
      {
       $u  = $y - 1700;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 8.83 + 0.1603*$u
                   - 0.0059285*$u2
                   + 0.00013336*$u3
                   - $u4/1174000;
      }


   if ($Y >= 1800 and $Y < 1860)
      {
       $u  = $y - 1800;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;
       $u7 = $u * $u6;

       return 13.72 - 0.332447*$u
                    + 0.0068612*$u2
                    + 0.0041116*$u3
                    - 0.00037436*$u4
                    + 0.0000121272*$u5
                    - 0.0000001699*$u6
                    + 0.000000000875*$u7;
      }


   if ($Y >= 1860 and $Y < 1900)
      {
       $u  = $y - 1860;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 7.62 + 0.5737*$u
                   - 0.251754*$u2
                   + 0.01680668*$u3
                   - 0.0004473624*$u4
                   + $u5 / 233174;
      }


   if ($Y >= 1900 and $Y < 1920)
      {
       $u  = $y - 1900;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 1.494119*$u - 2.79
                          - 0.0598939*$u2
                          + 0.0061966*$u3
                          - 0.000197*$u4;
      }


   if ($Y >= 1920 and $Y < 1941)
      {
       $u  = $y - 1920;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 21.20 + 0.84493*$u - 0.076100*$u2 + 0.0020936*$u3;
      }


   if ($Y >= 1941 and $Y < 1961)
      {
       $u  = $y - 1950;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 29.07 + 0.407*$u - $u2/233 + $u3/2547;
      }


   if ($Y >= 1961 and $Y < 1986)
      {
       $u = $y - 1975;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 45.45 + 1.067*$u - $u2/260 - $u3/718;
      }


   if ($Y >= 1986 and $Y < 2005)
      {
       $u  = $y - 2000;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 63.86 + 0.3345*$u
                    - 0.060374*$u2
                    + 0.0017275*$u3
                    + 0.000651814*$u4
                    + 0.00002373599*$u5;
      }


   if ($Y >= 2005 and $Y < 2050)
      {
       $u  = $y - 2000;

       return 62.92 + 0.32217*$u + 0.005589*$u*$u;
      }


   if ($Y >= 2050 and $Y <= 2150)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 0.5628*(2150 - $y) - 20;
      }


   if ($Y > 2150 and $Y <= 3000)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 20;
      }

   return $dT;

} // End of  Delta_T_Estimate(...)



?>


