<?php

/*
   This program displays a shaded lunar phase simulation of the near side
   based on a NASA/JPL lunar surface map from Clementine imaging and a
   digital elevation model (DEM) based on LOLA elevation altimetry data.

   Author   : Jay Tanner - 2017 - revised: 2025
   Language : PHP v5.6.11 - Revised with PHP 8.2.12
   License  : Public Domain

*/

   $HTMLTitle      = 'Western Lunar Hemisphere Phase Explorer - HD Edition';
   $InterfaceTitle = "<span style='color:white; font-size:135%; font-weight:normal;'>Western Lunar Hemisphere Phase Explorer - HD Edition<br></span><span style='font-size:80%;'><span style='font-size:80%;'>Earth is Directly to the Right</span></span><br>";
   $ProgramVersion = '';

   $out = '';

   $BgColor = "#222200";

// Point to this self-modifying program script.
   $RUN_THIS_PROGRAM = $_SERVER['SCRIPT_NAME'];

// Get current universal date/time (UTC)
// elements from the system clock.
   $Y  = gmdate('Y');
   $m  = gmdate('m');
   $d  = gmdate('d');
   $hh = gmdate('H');
   $mm = gmdate('i');
   $ss = gmdate('s');

// Compute JD00UTC for current date (UTC).
   $JD00UTC = GregorianToJD($m, $d, $Y) - 0.5;

// Compute fraction of day corresponding to UTC.
   $UTCFrac = (3600*$hh + 60*$mm + $ss) / 86400;

// Compute Delta T correction estimate as fraction of day.
   $DeltaTfrac = Delta_T_Estimate($Y, $m) / 86400;

// Add the time (UTC) fraction and apply the Delta T
// fraction to reduce it to the TT time scale and
// obtain the full astronomical JD number for the
// current UTC.
   $JDTT = $JD00UTC + $UTCFrac - $DeltaTfrac;



// Check if [Set To Current Phase] button was clicked
   if (@trim($_POST['current_button']) == 'Current Phase')
      {
//     Compute current geocentric lunar phase angle to nearest degree.
       $PhaseAngDeg = Lunar_Phase_Angle($JDTT);

//     Compute west side phase angle
       $PhaseAngDeg += 270;
       if($PhaseAngDeg >= 360) {$PhaseAngDeg -= 360;}
      }
else
     {
//    Read phase angle value from interface.
      $PhaseAngDeg = @trim($_POST['PhaseAngDeg']);
     }





// Check if [Compute] button was clicked.
   $compute_button = @trim($_POST['compute']);

// Check if [Invert Illumination] button was clicked.
   $invert_illumination_button = @trim($_POST['invert_illumination']);

// Check if [Mirror Phase] button was clicked.
   $mirror_phase_button = @trim($_POST['mirror_phase']);

// ------------------------------------------------------------
// If initial call, then display current lunar west side phase
// based on the computed JDTT value corresponding to the UTC
// derived from the system clock and Delta T computed from
// the NASA polynomials.
   if ($PhaseAngDeg == '')
      {
       $PhaseAngDeg = Lunar_Phase_Angle($JDTT) + 270;
       if($PhaseAngDeg >= 360) {$PhaseAngDeg -= 360;}
      }

// Round near side phase angle to nearest degree.
   $PhaseAngDeg = floor($PhaseAngDeg + 0.5);

// Check if [-] or [+] button was clicked.
// If so, then decrement or increment the
// phase angle accordingly. Otherwise, do
// nothing.

// Check if minus [-] button was clicked.
   if (@trim($_POST['minus_button'])   == "-") {$PhaseAngDeg--;}

// Check if plus [+] button was clicked.
   if (@trim($_POST['plus_button' ])   == '+') {$PhaseAngDeg++;}

// Check if [Set to Random Phase] button was clicked
   if (@trim($_POST['random_button' ]) == 'Random Phase')
      {
          if (mt_rand() % 2 == 0)
             {$PhaseAngDeg = mt_rand(20,  160);}
       else
             {$PhaseAngDeg = mt_rand(200, 340);}
      }

// Account for -1 degree value.
   if ($PhaseAngDeg < 0){$PhaseAngDeg += 360;}


// Account for 360 degree value.
   if ($PhaseAngDeg == 360){$PhaseAngDeg = 0;}


//    Check if [Invert Illumination] button was clicked.
//    If so, then add 180 degrees to phase angle and
//    modulate to fall in the range from 0 to 360.
      if ($invert_illumination_button == "Invert Illumination")
         {
          $PhaseAngDeg += 180;

          $PhaseAngDeg += ($PhaseAngDeg >= 360)? -360:0;
         }

   $out = "<img src='west-side-1500x1500/" . sprintf('%03d', $PhaseAngDeg) . '.webp' . "' alt=''>";

//     -----------------------------------------------
//     Check if [Mirror Phase] button was clicked.  If
//     so, then subtract phase angle from 360 degrees.

       if ($mirror_phase_button == "Mirror Phase")
          {
           $PhaseAngDeg = 360 - $PhaseAngDeg;
          }

   $out = "<img src='west-side-1500x1500/" . sprintf('%03d', $PhaseAngDeg) . '.webp' . "'  alt=''>";

// -----------------------------------------
// Error if phase angle outside valid range.

   if ($PhaseAngDeg < 0 or $PhaseAngDeg > 360)
      {
       $out = "<img src='west-side-1500x1500/" . "error.webp'" . "' alt=''>";
      }













// ---------------------------
// Generate HTML web page.

print <<< _HTML

<!DOCTYPE HTML>
<html>

<head>

<meta http-equiv="content-type" content="text/html; charset=UTF-8">
<meta name="description" content="Lunar Phase Explorer HD Edition">
<meta name="author" content="Jay Tanner">
<meta name='robots'    content='index,follow'>
<meta name='googlebot' content='index,follow'>
<meta http-equiv="pragma"  content="no-cache">
<meta http-equiv="expires" content="-1">
<meta name="robots"    content="index,follow">
<meta name="googlebot" content="index,follow">

<style type='text/css'>

BODY{font-family:Verdana; line-height: 120%;}

TD{padding:8px;}

INPUT[type="text"]:focus {font-family:monospace; font-size:175%; font-weight:bold; background-color:white; border:2px solid red;   text-align:center; border-radius:4px;}
INPUT[type="text"]       {font-family:monospace; font-size:175%; font-weight:bold; background-color:white; border:2px solid black; text-align:center; border-radius:4px;}
INPUT[type="text"]::-ms-clear {width:0; height:0;} /* Remove [x] from input MSIE text boxes */

INPUT[type='submit']{font-size:100%; font-weight:bold; letter-spacing:2px;}

INPUT[type='submit']:hover
     {font-size:100% font-weight:bold; background:cyan; border-radius:4px;}

A:link,  A:visited {background-color:transparent; color:LightBlue; font-family:Verdana; font-size:90%; font-weight:bold; text-decoration:none; line-height:150%; padding:2px;}
A:hover, A:active  {background-color:transparent; color:red;       font-family:Verdana; font-size:90%; font-weight:bold; text-decoration:none; line-height:150%; padding:2px;}
</style>

<title>Western Lunar Hemisphere Phase Explorer - HD Edition</title>

</head>


<body bgcolor='black' style='background-attachment:fixed; background-image:url("west-side-1500x1500/random-stars-background.webp");'>

<div ID='SmoothScroll'>

<table align="center" cellspacing='0' style='border:1px solid #444444; border-radius:8px 8px 0px 0px;'><tr><td>

<div align="center" style='background-color:black;'>

<div align="center" style='background-color:#1A1A00; padding:4px; color:white; font-size:80%; border:2px solid orange; border-radius:8px 8px 0px 0px;'>$InterfaceTitle<br>PHP Program and POV-Ray Graphics by Jay Tanner - $Y</div>
<form name="form1" method="post" action="$RUN_THIS_PROGRAM">

<span style='color:LightBlue; font-size:90%;'>
<br><b>Relative Phase Angle in Degrees</b> &nbsp;(0 &nbsp;to&nbsp; &plusmn;360):&nbsp;&nbsp;
<input style="text-align: center;" name="PhaseAngDeg" type="text" value="$PhaseAngDeg" size="4" maxlength="4">&nbsp;<b>&deg;</b>&nbsp;&nbsp;&nbsp;
</span>

<input name="compute_button" type="submit" value="Compute">
<br><br>
<input name="minus_button"        type="submit" value="-" title=' Decrement Phase by 1 Degree '>
<input name="plus_button"         type="submit" value="+" title=' Increment Phase by 1 Degree '>
<input name="mirror_phase"        type="submit" value="Mirror Phase">
<input name="invert_illumination" type="submit" value="Invert Illumination" title=' Swap Light / Dark '>
<input name="random_button"       type="submit" value="Random Phase">
<input name="current_button"      type="submit" value="Current Phase"><br><br>
</form>

<pre style='color:gray; font-family:Verdana; font-size:100%; font-weight:normal;'>Relative&nbsp;Phase&nbsp;Angles:&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;0=NM&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;90=FQ&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;180=FM&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;270=LQ</pre>

</div>

</td></tr></table>
<table ID='SmoothScroll' align='center'><tr><td align='center'><br><span style='color:DodgerBlue; font-size:90%'>N<br>&#124;</span><br>$out<br><span style='color:#FF6666; font-size:90%'>&#124;<br>S</span>
<br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br>
<br><br><br><br><br><br><br><br><br><br><br><br><br><br><br><br>
</td></tr>


</table>
</div>

</body>
</html>

_HTML;



/*
   A101.php

   THIS ASTRONOMY MODULE CONTAINS THE COMMON CORE
   LOGIC FOR THE LUNAR HEMISPHERIC PHASE EXPLORERS.
*/



/* ------------------------------------------------------
   Level 1

   DEPENDENCY:
   JD_to_Date_Time(...)
   Lunar_Phase_Table(...)
   Swap_LaGrange_Data(...)
   LaGrange_Interpolate(...)
   Deg_Mod_360(...)

   Given the astronomical JD number for any given moment,
   compute the corresponding geocentric lunar phase angle
   from 0 to +360 degrees.


*/

   function Lunar_Phase_Angle($JDTT)
{
// Get year/month substring for which to compute table.
   $Ym = substr(JD_to_Date_Time($JDTT), 0,6);

// Get 1st table: "PhaseAngDeg JD12TT"
   $PhaseTable1 = Lunar_Phase_Table($Ym);

// Create inverted 2nd table: "JD12TT PhaseAngDeg+
   $PhaseTable2 = Swap_LaGrange_Data($PhaseTable1);

// Given JD, interpolate the corresponding PhaseAngDeg
// within PhaseTable2
   $PhaseAngDeg = LaGrange_Interpolate($PhaseTable2, $JDTT);

// Modulate to fall in the range from 0 to 360 degrees.
   $PhaseAngDeg = sprintf("%1.5f", Deg_Mod_360($PhaseAngDeg));

// Done.
   return $PhaseAngDeg;
}






/* ------------------------------------------------------
   Level 1

   DEPENDENCY:
   File: (Lunar_Phases_1600_to_2200.data)
   Date_Time_to_JD(...)

*/
   function Lunar_Phase_Table($YmStr)
{
   global $DataPath;

   $LunarPhaseTable = "";

// Parse and extract the date elements.
   $Ym = trim($YmStr);
   $Y  = substr($Ym, 0,4);
   $m  = intval(substr($Ym, 4,2));

// Load lunar phases data file and count the file lines.
   $PhaseDataArray = file("../data/Lunar-Phases-1600-to-2200.data");
   $PhaseDataArrayCount = count($PhaseDataArray);

// ---------------------------------
// Compute JD12 for start/end dates.
   $JD12Start = Date_Time_to_JD($Ym . "01 12:00");

// Check for bad date and return FALSE if outside valid tabular range.
   if ($JD12Start === FALSE or $JD12Start < 2305448 or $JD12Start > 2524958)
       {return FALSE;}




// --------------------------------------------------------------
// Compute number of days in month, accounting for any leap year.
   $mDays = substr("312831303130313130313031", 2*($m-1), 2);

   if ($m == 2)
      {
       $mDays += (((!($Y % 4) and ($Y % 100)) or !($Y % 400))? 1:0);
      }


// ------------------------------
// Compute start/end JD12 values.
   $JD12Start -= 2;
   $JD12End = $JD12Start + $mDays + 3;

// --------------------------------------------------
// Search data array for JD12 start value and extract
// month sub-table equal to length of given month
// plus an extra day at each end of the month.

   for ($i=0;   $i < $PhaseDataArrayCount;   $i++)
  {
   $PhaseDataLine = trim($PhaseDataArray[$i]);

   if (intval($PhaseDataLine) > 0)
      {
//       list($PhaseAngDeg, $JD12i) = preg_split("[ ]", $PhaseDataLine);
$JD12i = substr($PhaseDataLine, -7);
$PhaseAngDeg = substr($PhaseDataLine, 0, strlen($PhaseDataLine)-7);


       if ($JD12i >= $JD12Start-1 and $JD12i <= $JD12End+1)
          {
           $LunarPhaseTable .= (($PhaseAngDeg/100000) . " $JD12i\n");
          }
      }
  }

// --------------------------------------------------------
// Get month table array into working array.  In this case,
// the data delimiters are the new-line (\n) breaks at the
// end of each text data line in the table.  Then adjust
// the phases to fall in the range from 0 to 810 degrees
// for subsequent interpolative continuity over the span.

   $wArray = preg_split("[\n]", trim($LunarPhaseTable));

   $wArrayCount = count($wArray);

   $LunarPhaseTable = "";

   $flag = $PrevPhase = FALSE;

   for ($j=1;   $j < $wArrayCount-1;   $j++)
  {
   list ($PhaseAngDeg, $JD12) = preg_split("[ ]", $wArray[$j]);

   if (floatval($wArray[$j]) - floatval($wArray[$j-1]) < 0 or $flag === TRUE)
  {
   $PhaseAngDeg += 360;
   $PhaseAngDeg += ($PrevPhase - $PhaseAngDeg > 0 and $flag === TRUE)? 360:0;
   $PrevPhase = $PhaseAngDeg;
   $flag = TRUE;
  }
   $PhaseAngDeg = sprintf("%1.5f", $PhaseAngDeg);
   $LunarPhaseTable .= "$PhaseAngDeg $JD12\n";
  }
   return trim($LunarPhaseTable);
}




   function Date_Time_to_JD ($Ymd_HMS)
{
// Strip all white space from date/time string.
   $w = preg_replace("/\s+/", " ", trim($Ymd_HMS));

// If no time string, then attach default zero time.
   if (strpos($w, " ") === FALSE) {$w .= " 00:00:00";}

// Split date/time string into separate variables.
   list($Ymd, $HMS) = preg_split("[ ]", $w);

// Convert HMS time elements string into HMS array and
// then count the colon-delimited time elements.
   $HMS = preg_split("[\:]", $HMS);
   $HMSCount = count($HMS);

// Fetch the individual time element (hh,mm,ss) values.
   $hh = ($HMSCount >= 1)? $HMS[0]:0;
   $mm = ($HMSCount >= 2)? $HMS[1]:0;
   $ss = ($HMSCount >= 3)? $HMS[2]:0;

// Parse the ISO integer-encoded date argument and extract
// the individual date arguments (Y,m,d) from within it.
   $Y = floor($Ymd / 10000);
   $m = floor(($Ymd - 10000*$Y) / 100);
   $d = $Ymd - 10000*$Y - 100*$m;

// Check date for validity (must be in the range from 1600 to 2400).
   if(checkdate($m,$d,$Y) === FALSE or $Y < 1600 or $Y > 2400) {return FALSE;}

// Compute Gregorian JD number for 12h TT of
// date and then the JD00 value from it.
   $JD00 = GregorianToJD($m, $d, $Y) - 0.5;

// Compute the JD fraction corresponding
// to the given time (hh,mm,ss) elements.
   $sh = bcmul("3600", $hh, 20);
   $sm = bcmul("60",   $mm, 20);
   $JDFrac = bcdiv(bcadd($sh, bcadd($sm,$ss, 20), 20),"86400", 20);

// Apply fraction of day to JD value and
// round off to up to 16 decimals max.
   $JD = bcadd(bcadd($JD00, $JDFrac, 20), "0.00000000000000005", 16);

// Trim off any redundant zeros and/or decimal point. Done.
   return rtrim(rtrim($JD, "0"), ".");

} // End of  Date_Time_to_JD(...)



/*
   =====================================================================
   Level 0:

   This function serves as an inverse Gregorian JD function.  Given the
   general JD number, this function returns the date and time resolved
   to the given number of decimals from 0 to 3.
   =====================================================================
*/

   function JD_to_Date_Time ($JDArg, $ssDecimals=0)
{
// Read JD number argument.
   $JD = trim($JDArg);

// Error if argument is non-numeric.
   if (!is_numeric($JD)) {return FALSE;}

// If JD is an integer value, then attach ".0" to end of it.
   if (strpos($JD, ".") === FALSE) {$JD .= ".0";}

// Compute JD12 value for date corresponding to JD argument.
   $JD12 = floor($JD + 0.5);

// Compute Gregorian date corresponding to the JD argument.
   list($m, $d, $Y) = preg_split("[\/]", JDtoGregorian($JD12));

// Force month and day values into 2-digit format.
   $m = sprintf("%02d", $m);
   $d = sprintf("%02d", $d);

// Construct integer encoded date from date elements.
   $Ymd = "$Y$m$d";

// Isolate fractional part of JD that will be converted into time.
   $j = bcsub($JD, "0.5", 16);
   $hours = bcmul("24", "0" . substr($j, strpos($j, "."), strlen($j)), 16);

// Compute hours value.
   $hh = floor($hours);

// Compute minutes value.
   $minutes = 60*($hours - $hh);
   $mm = floor($minutes);

// Compute seconds value
   $seconds = 60*($minutes - $mm);

// Format (hh, mm, ss) values for output.
   $hh = sprintf("%02d",  $hh);
   $mm = sprintf("%02d",  $mm);
   $ss = sprintf("%1.$ssDecimals" . "f",$seconds); if($ss < 10) {$ss = "0$ss";}

// Done.
   return "$Ymd $hh:$mm:$ss";

} // End of  JD_to_Date_Time(...)




/*
   ========================================================================
   Level 0

   This function performs LaGrange interpolation within an XY data table
   given as a string of matching space-delimited data pairs.

   ========================================================================
*/

   function LaGrange_Interpolate ($XYDataTable, $xArg)
{

// -----------
// Initialize.

   $XDataStr = $YDataStr = "";

// --------------------------------------------------------------
// Read and split XY data pairs into a work array.  In the array,
// even number indices = X-data, odd number indices = Y-data.

   $XY = preg_split("[ ]", preg_replace("/\s+/", " ", trim($XYDataTable)));

// ---------------------------------------------
// Count the total number of data elements. This
// value should always be an even number.

   $TotalDataCount = count($XY);

// ------------------------------------------------------------
// Number of data pairs.  This value should be an integer value
// exactly equal to 1/2 the total number of data points. If not,
// then there is an odd mismatched data point.

   $n = $TotalDataCount / 2;

// -----------------------------------------------------------
// Return error message if data vector element count mismatch.
// For every X value there must be a corresponding Y value or
// an XY data count mismatch error occurs.  An error will also
// occur if insufficient data.  There must be at least two XY
// data points given.

   if ($TotalDataCount < 4 )
      {return "ERROR: There must be at least two XY data pairs.";}

   if ($n != floor($n + 0.5))
      {return "ERROR: XY Data Count Mismatch. Odd data element.";}

// ------------------------------------------------------------
// Compute number of XY data pairs.  This value is exactly half
// the value of the total number of data elements.

   $n = $TotalDataCount / 2;

// -------------------------------------------------------
// Construct separate XY data strings from the array data.
// The XY data strings should each contain the same number
// of data elements.

   for($i=0;   $i < $TotalDataCount;   $i+=2)
      {
       $XDataStr .= $XY[$i]   . " ";
       $YDataStr .= $XY[$i+1] . " ";
      }

// --------------------------------------------------------------
// Split the created XY data vector strings into matching indexed
// arrays.  For every X value there must be a matching Y value
// and no two X values can be identical.

   $X = preg_split("[ ]", trim($XDataStr));
   $Y = preg_split("[ ]", trim($YDataStr));

// ----------------------------------------
// Read X argument for which to interpolate
// the Y value from the given XY data.

   $x = trim($xArg);  if ($x == "") {$x = "0";}

// -----------------------------------
// Initialize Y summation accumulator.
   $y = 0.0;

// -----------------------------------------------------
// Compute Lagrangian product (Li) for given X argument.

   for ($i=0;   $i < $n;   $i++)
       {
        $Li = 1.0;

        for ($j=0;   $j < $n;   $j++)
            {
             if ($j != $i) // Skip this cycle when j == i
                {
                 $Li = ($Li * ($x - $X[$j])) / ($X[$i] - $X[$j]);
                }
            } // Next j

//      --------------------------------------
//      Accumulate sum of Yi polynomial terms.

        $y += ($Y[$i] * $Li);

       } // Next i

   return $y;

} // End of  LaGrange_Interpolate(...)




/*
   ========================================================================
   Level 0

   This function simply swaps all matching XY data pairs held in a
   space delimited string argument. It was designed as a companion
   to the Lagrange_Interpolate() function to easily facilitate the
   swapping of the XY data interpolation roles.

   The data trend can be linear or non-linear, however
   only 2 data pairs are needed for any linear trend.

   ERRORS:
   On error FALSE is returned.  An error occurs if
   odd number of data elements or < 4 elements.
   ========================================================================
*/

   function Swap_LaGrange_Data ($XYDataStr)
{

// -------------------------------------------------------------------------
// Read space-delimited data string and split items into a sequential array.
   $XYArray = preg_split("[ ]", preg_replace("/\s+/", " ", trim($XYDataStr)));

   $XYArrayCount = count($XYArray);

   if ($XYArrayCount % 2 != 0 or $XYArrayCount < 4) {return FALSE;}

   $XYPairsCount = $XYArrayCount / 2;

// -------------------------------------------------------
// Construct and return output string of swapped XY pairs.
   $YXDataPairsStr = "";

   for ($i=1;   $i < $XYArrayCount;   $i+=2)
       {
        $YXDataPairsStr .= $XYArray[$i] . " " . $XYArray[$i-1] . "\n";
       }

   return trim($YXDataPairsStr);
}


/*
   ========================================================================
   Level 0

   This function modulates a negative or positive angle to fall within the
   range from 0 to +360 degrees to help handle angles that fall outside
   the standard circular values.  For example, -5 degrees would be
   converted to +355 degrees or 417 degrees would be converted
   to 57 degrees, etc.

   This function uses arbitrary precision arithmetic.

   ARGUMENTS:
   $DegStr = Degrees argument as a numerical string.

*/

   function Deg_Mod_360 ($DegStr)
  {
   $SignVal = ($DegStr < 0)? -1:1;

   $a = bcmul($SignVal, "$DegStr", 20);
   $b = bcmul($SignVal, bcsub($a, bcmul(360, bcdiv($a, 360)), 20),20);

   return rtrim(rtrim(bcadd($b, (($b < 0)? 360:0), 16), "0"), ".");

  } // End of  Deg_Mod_360()





/*
  ========================================================================
  Level 0

  This function computes the theoretical astronomical delta-T estimate
  in seconds for any dates between -2000 and 3000, based on the NASA
  polymomial expressions for Delta T

  NASA Ref:
  http://eclipse.gsfc.nasa.gov/LEcat5/deltatpoly.html


  ARGUMENTS:
  Y = Year number (-2000 BC to 3000 AD) (Neg=BC)(no year zero)
  mNum = Month number(1 to 12)

  ERRORS:
  FALSE is returned on error.  An error results if either
  the year or month number is invalid. A zero year value
  will return an error.
  ========================================================================
*/

   function Delta_T_Estimate ($Y, $mNum=1)
{

// --------------------------------------------------------
// The delta T value returns FALSE for invalid argument(s).

   $dT = FALSE;

// ----------------------------------------------
// ERROR if year value is non-numeric. It must be
// a number in the range from -2000 to +3000.

   if (!is_numeric($Y)) {return FALSE;}

// ------------------------------------------
// ERROR if year value is invalid. It must be
// in the range from -2000 to +3000.

   if ($Y < -2000 or $Y == 0 or $Y > 3000) {return FALSE;}

// -------------------------------------------------------
// ERROR if month value is outside the range from 1 to 12.
   if ($mNum < 1 or $mNum > 12) {return FALSE;}

// -----------------------------------------------------------
// If a negative (BC) year is given , it is converted into the
// corresponding mathematical year by adding 1.
//
// The year 100 BC (-100), becomes the mathematical year -99.

   $y = $Y + (($Y < 0)? 1:0) + ($mNum - 0.5)/12;


   if ($Y >= -2000 and $Y < -500)
      {
       $u  = ($Y - 1820) / 100;

       return 32*$u*$u - 20;
      }


   if ($Y >= -500 and $Y < 500)
      {
       $u  = $y/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 10583.6 - 1014.41*$u
                      +   33.78311*$u2
                      -    5.952053*$u3
                      -    0.1798452*$u4
                      +    0.022174192*$u5
                      +    0.0090316521*$u6;
      }


   if ($Y >= 500 and $Y < 1600)
      {
       $u  = ($y - 1000)/100;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;

       return 1574.2 - 556.01*$u
                     +  71.23472*$u2
                     +   0.319781*$u3
                     -   0.8503463*$u4
                     -   0.005050998*$u5
                     +   0.0083572073*$u6;
      }


   if ($Y >= 1600 and $Y < 1700)
      {
       $u  = $y - 1600;

       return 120 - 0.9808*$u - 0.01532*$u*$u + $u*$u*$u/7129;
      }


   if ($Y >= 1700 and $Y < 1800)
      {
       $u  = $y - 1700;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 8.83 + 0.1603*$u
                   - 0.0059285*$u2
                   + 0.00013336*$u3
                   - $u4/1174000;
      }


   if ($Y >= 1800 and $Y < 1860)
      {
       $u  = $y - 1800;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;
       $u6 = $u * $u5;
       $u7 = $u * $u6;

       return 13.72 - 0.332447*$u
                    + 0.0068612*$u2
                    + 0.0041116*$u3
                    - 0.00037436*$u4
                    + 0.0000121272*$u5
                    - 0.0000001699*$u6
                    + 0.000000000875*$u7;
      }


   if ($Y >= 1860 and $Y < 1900)
      {
       $u  = $y - 1860;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 7.62 + 0.5737*$u
                   - 0.251754*$u2
                   + 0.01680668*$u3
                   - 0.0004473624*$u4
                   + $u5 / 233174;
      }


   if ($Y >= 1900 and $Y < 1920)
      {
       $u  = $y - 1900;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;

       return 1.494119*$u - 2.79
                          - 0.0598939*$u2
                          + 0.0061966*$u3
                          - 0.000197*$u4;
      }


   if ($Y >= 1920 and $Y < 1941)
      {
       $u  = $y - 1920;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 21.20 + 0.84493*$u - 0.076100*$u2 + 0.0020936*$u3;
      }


   if ($Y >= 1941 and $Y < 1961)
      {
       $u  = $y - 1950;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 29.07 + 0.407*$u - $u2/233 + $u3/2547;
      }


   if ($Y >= 1961 and $Y < 1986)
      {
       $u = $y - 1975;
       $u2 = $u * $u;
       $u3 = $u * $u2;

       return 45.45 + 1.067*$u - $u2/260 - $u3/718;
      }


   if ($Y >= 1986 and $Y < 2005)
      {
       $u  = $y - 2000;
       $u2 = $u * $u;
       $u3 = $u * $u2;
       $u4 = $u * $u3;
       $u5 = $u * $u4;

       return 63.86 + 0.3345*$u
                    - 0.060374*$u2
                    + 0.0017275*$u3
                    + 0.000651814*$u4
                    + 0.00002373599*$u5;
      }


   if ($Y >= 2005 and $Y < 2050)
      {
       $u  = $y - 2000;

       return 62.92 + 0.32217*$u + 0.005589*$u*$u;
      }


   if ($Y >= 2050 and $Y <= 2150)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 0.5628*(2150 - $y) - 20;
      }


   if ($Y > 2150 and $Y <= 3000)
      {
       $u  = ($y - 1820) / 100;

       return 32*$u*$u - 20;
      }

   return $dT;

} // End of  Delta_T_Estimate(...)









?>

